

# EXAMPLE input paramters that should be generated by plugin:
#### Input parameters ##########################################################
#
## Input file names
#BaseDirecotry = "/tmp/test//"
#FileObjData = "/tmp/test//__NEW_ObjectData.csv/coloc_test_NEW_ObjectData.csv"
#FileObjColoc = "/tmp/test//__NEW_ObjectColoc.csv/coloc_test_NEW_ObjectColoc.csv"
#FileImgColoc = "/tmp/test//__NEW_ImageColoc.csv/coloc_test_NEW_ImageColoc.csv"
#
## Group properties
#NumberOfGroups = 1
#NumberImagesPerEachGroup = c(1)
#GroupNames = c("Condition 1 name")
#
## Channel properties
#NumberOfChannels = 2
#NamesOfChannels = c("channel 1 name", "channel 2 name")
#MinIntensitiesInChannels = c(0, 0)
#
#ChannelPairs = list(c(0,1))


# -------------- Optional parameters

# Thresholds to remove some objects or images from the analysis
MinObjSize = 0
MaxObjSize = 100000
MinNumOfObjects = 1
OverlapThreshold = 0.5 # overlap threshold for colocalization based on number of objects

# display ordering of the groups 
# also posisble to remove some conditions from the analysis e.g GroupOrder = c(1,3,4) will only perform analysis on conditions 1, 3 and 4
GroupOrder = seq(1, NumberOfGroups)

MaxColocYValue=75  # maximum Y display axis

# Output directories and file names
OutputDirectory = "new_colocalization_analysis"
OutputDataDirectory = "data"

#size ranges parameters (should be sequences of 4 values each)
#to test sequence of threholds for size and intensity effect
SizeThresholdsRange = seq(2, 20, length.out = 4)
IntensityThresholdRange = seq(0.05,0.5, length.out = 4)

### Functions #################################################################

debugPrint = function(str) {
    # set TRUE or FALSE to turn on/off debug messages
    if (TRUE) {
        print(str)
    }
}

quitNow = function(str) {
  debugPrint(str)
  quit(status = 1)
}

# Assigned multiple value to variables like: getMultiValue[q,w,r]=c(1,2,3)
getMultiValue <- structure(NA, class="result") 
"[<-.result" <- function(x, ..., value) {
  args <- as.list(match.call())
  args <- args[-c(1:2,length(args))]
  length(value) <- length(args)
  for(i in seq(along=args)) {
    a <- args[[i]]
    if(!missing(a)) eval.parent(substitute(a <- v,list(a=a,v=value[[i]])))
  }
  x
}

# Returns data frame with means/sem for every group and matrix with coloc number based for each image and group (groups are columns)
coloc = function(aFileNames, aData, aNumberImagesPerEachGroup, aNumberOfGroups, aChannelNum, aChannelNumColoc, aMinIntensitiesInChannels, aOverlapThreshold, aMinObjSize, aMaxObjSize, aMinNumOfObjects) {
    rd0 = matrix(nrow=max(aNumberImagesPerEachGroup), ncol=aNumberOfGroups)
    meanInGroup = 1;
    standardErrorOfMean = 1
    
    groupShift=0
    
    for(group in 1:aNumberOfGroups) {
        currentGroup=aData[[group]]
        rd=matrix(nrow=aNumberImagesPerEachGroup[group], ncol=1)
        
        for( imgIdx in 1:aNumberImagesPerEachGroup[group] ) {
            currentFileName = aFileNames[imgIdx + groupShift]
            n1 = nrow(currentGroup[
                    currentGroup$FileName==currentFileName &
                    currentGroup$Channel==aChannelNum & 
                    currentGroup$Intensity>aMinIntensitiesInChannels[aChannelNum + 1] & 
                    currentGroup$Size>aMinObjSize & 
                    currentGroup$Size<aMaxObjSize,])
            n2 = nrow(currentGroup[
                    currentGroup$FileName==currentFileName &
                    currentGroup$Channel==aChannelNumColoc & 
                    currentGroup$Intensity>aMinIntensitiesInChannels[aChannelNumColoc  + 1] & 
                    currentGroup$Size>aMinObjSize & 
                    currentGroup$Size<aMaxObjSize,])
            c1 = nrow(currentGroup[
                    currentGroup$FileName==currentFileName &
                    currentGroup$Channel==aChannelNum &
                    currentGroup$Intensity>aMinIntensitiesInChannels[aChannelNum  + 1] & 
                    currentGroup$Size>aMinObjSize & 
                    currentGroup$Size<aMaxObjSize & 
                    currentGroup$Overlap>aOverlapThreshold & 
                    currentGroup$AvgColocObjIntensity>aMinIntensitiesInChannels[aChannelNumColoc  + 1] & 
                    currentGroup$AvgColocObjSize<aMaxObjSize & 
                    currentGroup$AvgColocObjSize>aMinObjSize,])
            
            if (n1 < aMinNumOfObjects || n2 < aMinNumOfObjects) {
                rd[imgIdx, 1] = NA
            }	 
            else{
                rd[imgIdx, 1] = c1/n1
            }
            rd0[imgIdx, group] = rd[imgIdx, 1]
        }
        
        groupShift = groupShift + aNumberImagesPerEachGroup[group]
        
        meanInGroup[group] = mean(rd[,1], na.rm =TRUE)
        standardErrorOfMean[group] = sd(rd[,1], na.rm =TRUE) / sqrt(length(rd[,1]))
    }
    
    result = data.frame(mean=meanInGroup, sem=standardErrorOfMean)
    
    return(list(result, rd0))
}

# Returns data frame with means/sem for every group and matrix with coloc size or size/intensity based for each image and group (groups are columns)
COLOC_SIZE=1
COLOC_SIZE_AND_INTENSITY=2
colocSizeIntensity = function(aFileNames, aData, aNumberImagesPerEachGroup, aNumberOfGroups, aChannelNum, aChannelNumColoc, aMinIntensitiesInChannels, aMinObjSize, aMaxObjSize, aMinNumOfObjects, aType) {
    rd0 = matrix(nrow=max(aNumberImagesPerEachGroup), ncol=aNumberOfGroups)
    m=1
    s=1
    coInt=0
    tInt=0
    sum=0
    
    groupShift=0
    
    for(group in 1:aNumberOfGroups) {
        currentGroup=aData[[group]]
        rd=matrix(nrow=aNumberImagesPerEachGroup[group], ncol=1)
        j=1
        for( imgIdx in 1:aNumberImagesPerEachGroup[group] ) {
            currentFileName = aFileNames[imgIdx + groupShift]
            dtemp = currentGroup[
                    currentGroup$FileName==currentFileName &
                    currentGroup$Channel==aChannelNum & 
                    currentGroup$Intensity>aMinIntensitiesInChannels[aChannelNum + 1] & 
                    currentGroup$Size>aMinObjSize & 
                    currentGroup$Size<aMaxObjSize,]
            n1 = nrow(dtemp)
            nr = nrow(currentGroup[
                    currentGroup$FileName==currentFileName &
                    currentGroup$Channel==aChannelNumColoc & 
                    currentGroup$Intensity>aMinIntensitiesInChannels[aChannelNumColoc  + 1] & 
                    currentGroup$Size>aMinObjSize & 
                    currentGroup$Size<aMaxObjSize,])
            
            coInt=0
            tInt=0
            if(n1!=0){
                ov=0
                for(ves in 1:n1){
                    if(dtemp[ves,'AvgColocObjSize'] > aMaxObjSize || dtemp[ves,'AvgColocObjSize'] < aMinObjSize || dtemp[ves,'AvgColocObjIntensity'] < aMinIntensitiesInChannels[aChannelNumColoc  + 1] ){
                        ov=0
                    }else{ ov = dtemp[ves,'Overlap']} #objects colocalizing with non valid objects are considered non colocalizing
                    
                    switch(aType,
                           {
                               coInt=coInt + ov * dtemp[ves,'Size']
                               tInt=tInt+ dtemp[ves,'Size']
                           },
                           {
                               coInt=coInt+ ov * dtemp[ves,'Size'] * dtemp[ves,'Intensity']
                               tInt=tInt+ dtemp[ves,'Size'] * dtemp[ves,'Intensity']
                           })
                }
                res=coInt/tInt
            }
            else {
                res=0
            }
            if(nr>aMinNumOfObjects && n1>aMinNumOfObjects){
                rd[j,1]=res
                rd0[j,group]=res
                j=j+1
            }
        }
        
        groupShift = groupShift + aNumberImagesPerEachGroup[group]
        
        sum=sum+aNumberImagesPerEachGroup[group]
        m[group]=mean(rd[1:(j-1)],na.rm =TRUE)
        s[group]=sd(rd[1:(j-1)],na.rm =TRUE)/(sqrt(length(rd[1:(j-1)])))
    }
    
    result = data.frame(mean=m, sem=s)
    
    return(list(result, rd0))
}

### Pearson numbers
# Output Types:
PEARSON_CORR=1
PEARSON_CORR_WITH_MASK=2
pearson = function(aFileNames, aImageColocationData, aNumberOfGroups, aNumberImagesPerEachGroup, aChannelNum, aChannelNumColoc, aType) {
    m=1
    s=1
    rd0=matrix(nrow=max(aNumberImagesPerEachGroup), ncol=aNumberOfGroups)
    columnName=switch(aType, 'PearsonCorr', 'PearsonCorrMasked')
    groupShift=0
    for (group in 1:aNumberOfGroups) {
        rd=matrix(nrow=aNumberImagesPerEachGroup[group], ncol=1)
        for (imgIdx in 1:aNumberImagesPerEachGroup[group]) {
            currentFileName = aFileNames[imgIdx + groupShift]
            dtemp = aImageColocationData[
                    aImageColocationData$FileName==currentFileName &
                    aImageColocationData$Channel==aChannelNum & 
                    aImageColocationData$ChannelColoc==aChannelNumColoc, columnName]
            
            rd[imgIdx, 1] = dtemp
            rd0[imgIdx, group]=rd[imgIdx, 1]
        }
        groupShift = groupShift + aNumberImagesPerEachGroup[group]
        
        m[group]=mean(rd[,1],na.rm =TRUE)
        s[group]=sd(rd[,1],na.rm =TRUE)/(sqrt(length(rd[,1])))
    }
    
    result = data.frame(mean=m, sem=s)
    return(list(result, rd0))
}

pvstr = function(pval){
    if (is.nan(pval)) {pstring=sprintf("Nan")}
    else if (pval < 0.0001){pstring=sprintf("< 1e-4, ****" )}
    else if (pval < 0.001){pstring=sprintf("%.2e, ***",pval )}
    else if (pval < 0.01){pstring=sprintf("%.2e, **",pval )}
    else if (pval < 0.05){pstring=sprintf("%.2e, *",pval )}
    else {pstring= sprintf("%.2e, ns",pval )}
    
    return(pstring)
}

stat_aov_tukey= function(data){
    var = as.vector(data)
    cond = rep (GroupNames[GroupOrder],rep(max(NumberImagesPerEachGroup),length(GroupOrder)))
    data2= data.frame(var, cond) 
    aovr = aov(var ~ cond , data2) #1 way anova 
    tuk = TukeyHSD(aovr)# tukey test
    
    pval = summary(aovr)[[1]][1,"Pr(>F)"]
    pstring=pvstr(pval)
    
    return(list(pstring, pval, aovr, tuk))
} 

errorBar = function(x, y, upper, lower=upper, length=0.1,...){
    if(length(x) != length(y) | length(y) !=length(lower) | length(lower) != length(upper))
        stop("vectors must be same length")
    if( !is.na(max(upper)) ){
        suppressWarnings(arrows(x,y+upper, x, y-lower, angle=90, code=3, length=length, ...))
    }
}

plotresnumbers_min = function(res, xtitle, ytitle, ymax, ymin = 0, scale = 1) {
    ncolors=nrow(res$mean)
    if (is.null(ncolors)) {
        ncolors = 1
    }
    if (is.na(ymax) | is.na(ymin)) {
        barx <- barplot(scale*res$mean, beside=TRUE, names.arg=GroupNames[GroupOrder], col=rainbow(ncolors), axis.lty=1, xlab=xtitle, ylab=ytitle)
    } else {
        barx <- barplot(scale*res$mean, beside=TRUE, names.arg=GroupNames[GroupOrder], col=rainbow(ncolors), ylim=c(ymin,ymax), axis.lty=1, xlab=xtitle, ylab=ytitle)
    }
    errorBar(barx,scale*res$mean, scale*res$sem)
}

plotresf=function(means,sems, teffect, ymax, ylabel){
    barx <- barplot(100*means, beside=TRUE, ylim=c(0,ymax), names.arg=GroupNames[GroupOrder], axis.lty=1, xlab=teffect, ylab=ylabel)
    errorBar(barx,100*means,100*sems)
}

generateData = function(resc, res_imgs, aRescName, aResImgsName, aPlotName, aChannelNum, aChannelNumColoc, aChartName) {
    order= paste0("_", NamesOfChannels[aChannelNum + 1], "_in_", NamesOfChannels[aChannelNumColoc + 1])
    write.table(resc, file=paste0(outputDataPath,aRescName, order, ".csv"), quote = FALSE, row.names=GroupNames, col.names=c("GroupNames,mean", "sem"), sep=",")
    namesWithId = GroupNames
    namesWithId[1] = paste0("ImageID,", namesWithId[1])
    write.table(res_imgs, file=paste0(outputDataPath,aResImgsName, order, ".csv"), quote = FALSE, row.names = TRUE, col.names = namesWithId, sep = ",")
    if (NumberOfGroups > 1 && min(NumberImagesPerEachGroup) > 1) {
        getMultiValue[ps, p, a, t] = stat_aov_tukey(res_imgs[,GroupOrder])
    } else {
        ps = NA
    }
    plotresnumbers_min(resc[GroupOrder,], paste0(aPlotName,", P value ", ps), aChartName, MaxColocYValue, 0, 100)
    
    return(t)
}

generateColocNumber = function(aChannelNum, aChannelNumColoc, aChartName) {
    getMultiValue[resc, res_imgs] = coloc(uniqueNames, dataGroups, NumberImagesPerEachGroup, NumberOfGroups, aChannelNum, aChannelNumColoc, MinIntensitiesInChannels, OverlapThreshold, MinObjSize, MaxObjSize, MinNumOfObjects)
    t = generateData(resc, res_imgs, "Colocalization_number", "Colocalization_images_data_number", "Object number colocalization", aChannelNum, aChannelNumColoc, aChartName)
    return(t)
}

generateColocSize=function(aChannelNum, aChannelNumColoc, aChartName) {
    getMultiValue[resc, res_imgs] = colocSizeIntensity(uniqueNames, dataGroups, NumberImagesPerEachGroup, NumberOfGroups, aChannelNum, aChannelNumColoc, MinIntensitiesInChannels, MinObjSize, MaxObjSize, MinNumOfObjects, COLOC_SIZE)
    t = generateData(resc, res_imgs, "Colocalization_size", "Colocalization_images_data_size", "Size colocalization", aChannelNum, aChannelNumColoc, aChartName)
    return(t) 
}

generateColocEffect=function(aChannelNum, aChannelNumColoc, aChartName, aMinSize=MinObjSize) {
    getMultiValue[resc, res_imgs] = colocSizeIntensity(uniqueNames, dataGroups, NumberImagesPerEachGroup, NumberOfGroups, aChannelNum, aChannelNumColoc, MinIntensitiesInChannels, aMinSize, MaxObjSize, MinNumOfObjects, COLOC_SIZE_AND_INTENSITY)
    t = generateData(resc, res_imgs, "Colocalization_signal", "Colocalization_images_data_signal", "Signal colocalization (size and intensity)", aChannelNum, aChannelNumColoc, aChartName)
    return(t)
}

generatePearson=function(aChannelNum, aChannelNumColoc) {
    getMultiValue[resn,res_imgs] = pearson(uniqueNames, imageColocationData, NumberOfGroups, NumberImagesPerEachGroup, aChannelNum, aChannelNumColoc, PEARSON_CORR)
    if (NumberOfGroups > 1 && min(NumberImagesPerEachGroup) > 1) {
        getMultiValue[ps1, p, a, t1] = stat_aov_tukey(res_imgs[, GroupOrder])
    } else {
        ps1 = NA
    }
    
    getMultiValue[resn2,res_imgs] = pearson(uniqueNames, imageColocationData, NumberOfGroups, NumberImagesPerEachGroup, aChannelNum, aChannelNumColoc, PEARSON_CORR_WITH_MASK)
    if (NumberOfGroups > 1 && min(NumberImagesPerEachGroup) > 1) {
        getMultiValue[ps2, p, a, t2] = stat_aov_tukey(res_imgs[, GroupOrder]) 
    } else {
        ps2 = NA
    }
    
    minval= min(resn,resn2)
    if (!is.na(minval) && minval < 0) mindisp=-1 else mindisp=0
    plotresnumbers_min(resn[GroupOrder,],paste0("Pearson correlation, original image",", P value ", ps1), "Pearson correlation", MaxColocYValue/100,mindisp )
    plotresnumbers_min(resn2[GroupOrder,],paste0("Pearson correlation with cell masks\n and background removal",", P value ", ps2), " Pearson correlation", MaxColocYValue/100,mindisp )
    
    return(list(t1, t2))
}

THR_SIZE=1
THR_INTENSITY=2
generateThreshold = function(aChannelNum, aChannelNumColoc, aChartName, aRange, aType, aDescription) {
    order= paste0("_", NamesOfChannels[ChannelNum + 1], "_in_", NamesOfChannels[ChannelNumColoc + 1])
    resmeans=NULL
    ressems=NULL
    for(tMinSize in aRange){
        getMultiValue[t5,]=
            switch (aType, 
                    colocSizeIntensity(uniqueNames, dataGroups, NumberImagesPerEachGroup, NumberOfGroups, aChannelNum, aChannelNumColoc, MinIntensitiesInChannels, tMinSize, MaxObjSize, MinNumOfObjects, COLOC_SIZE_AND_INTENSITY),
                    {   
                        minInt=rep(tMinSize, NumberOfChannels)
                        colocSizeIntensity(uniqueNames, dataGroups, NumberImagesPerEachGroup, NumberOfGroups, aChannelNum, aChannelNumColoc, minInt, MinObjSize, MaxObjSize, MinNumOfObjects, COLOC_SIZE_AND_INTENSITY)
                    })
        resmeans = c(resmeans,t5$mean)
        ressems  = c(ressems, t5$sem)
    }
    means = matrix(resmeans,length(aRange), NumberOfGroups, byrow=TRUE)[,GroupOrder]
    sems = matrix(ressems,length(aRange), NumberOfGroups, byrow=TRUE)[,GroupOrder]
    plotresnumbers_min(list(mean=means, sem=sems), paste0(aDescription, ": ", paste(aRange, collapse = ' '),order), aChartName, MaxColocYValue, 0, 100)
}

tukeyplot = function (x, minp, maxp, xlabel, aFactor = 1) {
    for (i in seq_along(x)) {
        xi <- aFactor*x[[i]][, -4, drop = FALSE]
        yvals <- nrow(xi):1
        dev.hold()
        on.exit(dev.flush())
        plot(c(xi[, "lwr"], xi[, "upr"]), rep.int(yvals, 2), type = "n", axes = FALSE, xlab = xlabel, ylab = "", main = "", xlim=c(aFactor*minp,aFactor*maxp))
        axis(1)
        ynames=dimnames(xi)[[1L]]
        ypvalues=sapply(x$cond[,4], pvstr)
        ylabs=paste0(ynames,", ", ypvalues)
        axis(2, at = nrow(xi):1, labels = ylabs, srt = 0)
        abline(h = yvals, lty = 1, lwd = 0.5, col = "lightgray")
        abline(v = 0, lty = 2, lwd = 0.5)
        segments(xi[, "lwr"], yvals, xi[, "upr"], yvals)
        segments(as.vector(xi), rep.int(yvals - 0.1, 3), as.vector(xi), rep.int(yvals + 0.1, 3))
        box()
    }
}

### Different stats calcualtion
# Output Types:
VES_NUMBER=1
VES_SIZE=2
VES_TOTAL_SIZE=3
VES_SIZE_RATIO=4
VES_LENGHT=5
VES_INTENSITY=6
calculateStat = function(aFileNames, aData, aNumberImagesPerEachGroup, aNumberOfGroups, aChannelNum, aChannelNumColoc, aMinObjSize, aMaxObjSize, aType) {
    m1=1
    s1=1
    m2=1
    s2=1
    
    groupShift=0
    
    for(group in 1:aNumberOfGroups){
        currentGroup=aData[[group]]
        rd1=matrix(nrow=aNumberImagesPerEachGroup[group], ncol=1)
        rd2=matrix(nrow=aNumberImagesPerEachGroup[group], ncol=1)
        
        for(imgIdx in 1:aNumberImagesPerEachGroup[group]) {
            currentFileName = aFileNames[imgIdx + groupShift]
            dtemp1 = currentGroup[
                    currentGroup$FileName == currentFileName &
                    currentGroup$Channel == aChannelNum & 
                    currentGroup$ChannelColoc == aChannelNumColoc &
                    currentGroup$Size>aMinObjSize & 
                    currentGroup$Size<aMaxObjSize,]
            dtemp2 = currentGroup[
                    currentGroup$FileName == currentFileName &
                    currentGroup$Channel == aChannelNumColoc & 
                    currentGroup$ChannelColoc == aChannelNum &
                    currentGroup$Size>aMinObjSize & 
                    currentGroup$Size<aMaxObjSize,]
            rd1[imgIdx, 1] = switch(aType, 
                                   {nrow(dtemp1)}, 
                                   {mean(dtemp1$Size, na.rm =TRUE)},
                                   {sum(dtemp1$Size)},
                                   {sum(dtemp1$Size)/sum(dtemp2$Size)},
                                   {mean(dtemp1$Length, na.rm =TRUE)},
                                   {mean(dtemp1$Intensity, na.rm =TRUE)})
            rd2[imgIdx, 1] = switch(aType, 
                                    {nrow(dtemp2)}, 
                                    {mean(dtemp2$Size, na.rm =TRUE)},
                                    {sum(dtemp2$Size)},
                                    {sum(dtemp2$Size)/sum(dtemp1$Size)},
                                    {mean(dtemp2$Length, na.rm =TRUE)},
                                    {mean(dtemp2$Intensity, na.rm =TRUE)})
        }
        
        groupShift = groupShift + aNumberImagesPerEachGroup[group]
        
        m1[group] = mean(rd1[,1], na.rm = TRUE)
        s1[group] = sd(rd1[,1], na.rm = TRUE) / sqrt(length(rd1[,1]))
        m2[group] = mean(rd2[,1], na.rm = TRUE)
        s2[group] = sd(rd2[,1], na.rm = TRUE) / sqrt(length(rd2[,1]))
    }
    
    result1 = data.frame(mean=m1, sem=s1)
    result2 = data.frame(mean=m2, sem=s2)
    
    return(list(result1, result2))
}

###############################################################################
###############################################################################
###############################################################################

### Verify Input Params #######################################################
if (NumberOfGroups != length(NumberImagesPerEachGroup)) { quitNow("Wrong Dimensions! NumberImagesPerEachGroup") }
if (NumberOfGroups != length(GroupNames)) { quitNow("Wrong Dimensions! GroupNames") }
if (NumberOfChannels != length(NamesOfChannels)) { quitNow("Wrong Dimensions! NamesOfChannels") }
if (NumberOfChannels != length(MinIntensitiesInChannels)) { quitNow("Wrong Dimensions! MinIntensitiesInChannels") }
if (min(rapply(ChannelPairs, length)) != 2 | max(rapply(ChannelPairs, length)) != 2) { quitNow("Wrong channel pairs! (each pair should contain 2 numbers only )") }

### Load data from files and merge ############################################
separatorChar = ";"
commentChar = "%"
debugPrint(paste0("Reading Object coloc: ", FileObjColoc))
objectColocationData =read.csv(FileObjColoc, sep = separatorChar, comment.char=commentChar)
debugPrint(paste0("Reading Object data:  ", FileObjData))
objectData =read.csv(FileObjData, sep = separatorChar, comment.char=commentChar)
debugPrint(paste0("Reading Image coloc:  ", FileImgColoc))
imageColocationData =read.csv(FileImgColoc, sep = separatorChar, comment.char=commentChar)
# Object coloc and data are easy to merge - they have same number of rows and few sharing columns
mc=merge(objectColocationData, objectData)
# Here we have to give explicit way of merging. FileName...ChannelColoc fully identify row in Object*.csv
keyColumnsNames=c("FileName", "Frame", "Channel", "ChannelColoc")
data=merge(mc, imageColocationData, by=keyColumnsNames)

uniqueNames=unique(data$FileName)
if (length(uniqueNames) != sum(NumberImagesPerEachGroup)) { quitNow(paste0("Not enough images in the data: ", nrow(imageColocationData), " vs. ",  sum(NumberImagesPerEachGroup))); }

isData3D = !(max(data$Z) == 0 & min(data$Z) == 0)
debugPrint(paste0("Is data 3d: ", isData3D))

# Group input data into groups according to configured number of gorups and files in particular group
dataGroups = NULL
index = 1
for (group in 1:NumberOfGroups) {
  correctRows=seq(FALSE, FALSE, length.out = length(data$FileName))
  for (i in 1:NumberImagesPerEachGroup[group]) {
    correctRows=correctRows | data$FileName==uniqueNames[index]
    index = index + 1;
  }
  dataGroups = c(dataGroups, list(data[correctRows,]))
}

### Prepare Directories #######################################################
outputPath = paste0(BaseDirecotry, OutputDirectory, .Platform$file.sep)
debugPrint(paste0("Output path:      [", outputPath, "]"))
if (!dir.exists(outputPath)) dir.create(outputPath)
outputDataPath = paste0(outputPath, OutputDataDirectory, .Platform$file.sep)
debugPrint(paste0("Output data path: [", outputDataPath, "]"))
if (!dir.exists(outputDataPath)) dir.create(outputDataPath)

### Generate output PDFs #######################################################
#Iterate over all channel pairs and generate data
for (channelPair in ChannelPairs) {
    ChannelNum = channelPair[1]
    ChannelNumColoc = channelPair[2]
    channelsStr = paste0(ChannelNum, "_vs_", ChannelNumColoc)
    
    scriptInputParamsStr = sprintf("R script analysis parameters: MinIntCh1 %.3f MinIntCh2 %.3f MaxSize %d MinSize %d MinObjects %d", MinIntensitiesInChannels[ChannelNum + 1], MinIntensitiesInChannels[ChannelNumColoc + 1], MaxObjSize, MinObjSize, MinNumOfObjects)
    
    ### Generate Colocalization ###################################################
    debugPrint(paste0("++++++++++ COLOCALIZATION ++++++++++++++++ channels: ", channelsStr))
    ColocABname=paste0(" ( ", NamesOfChannels[ChannelNum + 1]," + ", NamesOfChannels[ChannelNumColoc + 1]," ) ", " / ", NamesOfChannels[ChannelNum + 1])
    ColocBAname=paste0(" ( ", NamesOfChannels[ChannelNum + 1]," + ", NamesOfChannels[ChannelNumColoc + 1]," ) ", " / ", NamesOfChannels[ChannelNumColoc + 1])
    colocPdfName=paste0(outputPath, "Colocalization_", channelsStr, ".pdf")
    debugPrint(paste0("Output PDF: ", colocPdfName))
    pdf(colocPdfName, width=8.2, height=11.3)
    par(mfrow=c(4,2), mar = c(4, 4, 1, 1) + 0.1, oma = c(5,0,2.5,0.2))
    
    t1 = generateColocNumber(ChannelNum, ChannelNumColoc, ColocABname)
    t2 = generateColocNumber(ChannelNumColoc, ChannelNum, ColocBAname)
    t3 = generateColocSize(ChannelNum, ChannelNumColoc, ColocABname)
    t4 = generateColocSize(ChannelNumColoc, ChannelNum, ColocBAname)
    t5 = generateColocEffect(ChannelNum, ChannelNumColoc, ColocABname)
    t6 = generateColocEffect(ChannelNumColoc, ChannelNum, ColocBAname)
    getMultiValue[t7, t8] = generatePearson(ChannelNum, ChannelNumColoc)
    
    mtext(paste0("Colocalization, ", min(NumberImagesPerEachGroup), " samples per group"), 3, line=0, adj=0.5, cex=1.2, outer=TRUE)
    mtext(as.character(scriptInputParamsStr), 1, line=3, adj=0, cex=0.55, outer=TRUE)
    mtext("Mean and sem displayed, P values from one way ANOVA, **** for p value < 0.0001, ***  0.0001 < p value < 0.001, **  0.001 < p value < 0.01, *  0.01 < p value < 0.05", 1, line=4, adj=0, cex=0.55, outer=TRUE)
    mtext(format(Sys.time(), "%Y-%m-%d "), cex=0.75, line=4, side=SOUTH<-1, adj=1, outer=TRUE)
    dev.off()
    
    ### Generate Tukey Plots ##################################################
    if (NumberOfGroups > 1 && min(NumberImagesPerEachGroup) > 1) {
        debugPrint(paste0("++++++++++ TUKEY PLOTS +++++++++++++++++++ channels: ", channelsStr))
        colocCiPdfName=paste0(outputPath, "ColocalizationCI_", channelsStr, ".pdf")
        debugPrint(paste0("Output PDF: ", colocCiPdfName))
        pdf(colocCiPdfName, width=8.2, height=11.3)
        par(mfrow=c(4,2), mar= c(4, 12, 1, 0.5) + 0.1, oma =c(5,0,2.5,0.2), las=1 )
        
        minp=min(t1$cond[,'lwr'], t2$cond[,'lwr'], t3$cond[,'lwr'], t4$cond[,'lwr'], t5$cond[,'lwr'], t6$cond[,'lwr'], t7$cond[,'lwr'], t8$cond[,'lwr'])
        maxp=max(t1$cond[,'upr'], t2$cond[,'upr'], t3$cond[,'upr'], t4$cond[,'upr'], t5$cond[,'upr'], t6$cond[,'upr'], t7$cond[,'upr'], t8$cond[,'upr'])
        
        tukeyplot(t1, minp, maxp, "Object number colocalization", 100)
        tukeyplot(t2, minp, maxp, "Object number colocalization", 100)
        tukeyplot(t3, minp, maxp, "Size colocalization", 100)
        tukeyplot(t4, minp, maxp, "Size colocalization", 100)
        tukeyplot(t5, minp, maxp, "Signal colocalization", 100)
        tukeyplot(t6, minp, maxp, "Signal colocalization", 100)
        tukeyplot(t7, minp, maxp, "Pearson")
        tukeyplot(t8, minp, maxp, "Pearson with mask and background removal")
        
        mtext(paste0("Tukey test, 95% family-wise confidence levels and P values, ", min(NumberImagesPerEachGroup), " samples per group"), 3, line=0, adj=0.5, cex=1.2, outer=TRUE)
        mtext(as.character(scriptInputParamsStr), 1, line=3, adj=0, cex=0.55, outer=TRUE)
        mtext("P values and confidence intervals from post-hoc Tukey test, **** for p value < 0.0001, ***  0.0001 < p value < 0.001, **  0.001 < p value < 0.01, *  0.01 < p value < 0.05", 1, line=4, adj=0, cex=0.55, outer=TRUE)
        mtext(format(Sys.time(), "%Y-%m-%d "), cex=0.75, line=4, side=SOUTH<-1, adj=1, outer=TRUE)
        
        dev.off()
    }
    
    ### Generate Secondary_results_1 ###################################################
    debugPrint(paste0("++++++++++ Secondary_results A +++++++++++++++++++ channels: ", channelsStr))
    colocObjectResultsPdfName=paste0(outputPath, "Secondary_resultsA_", channelsStr, ".pdf")
    debugPrint(paste0("Output PDF: ", colocObjectResultsPdfName))
    pdf(colocObjectResultsPdfName, width=8.2, height=11.3)
    par(mfrow=c(4,2), mar= c(4, 4, 1, 1) + 0.1, oma =c(4,0,1.5,0.2) )
    
    getMultiValue[resn, resn2]=calculateStat(uniqueNames, dataGroups, NumberImagesPerEachGroup, NumberOfGroups, ChannelNum, ChannelNumColoc, MinObjSize, MaxObjSize, VES_NUMBER)   
    maxdisp = max(resn[1]+resn[2], resn2[1]+resn2[2])
    if(is.na(maxdisp)) maxdisp=1.1*max(resn[1], resn2[1])
    plotresnumbers_min(resn[GroupOrder,],paste0(NamesOfChannels[ChannelNum + 1] ," Object Number"), "Object #", maxdisp)
    plotresnumbers_min(resn2[GroupOrder,],paste0(NamesOfChannels[ChannelNumColoc + 1] ," Object Number"), "Object #", maxdisp)
    
    getMultiValue[resv, resv2]=calculateStat(uniqueNames, dataGroups, NumberImagesPerEachGroup, NumberOfGroups, ChannelNum, ChannelNumColoc, MinObjSize, MaxObjSize, VES_SIZE)   
    if (isData3D){lbl="Volume [pixels^3]"}else{lbl="Area [pixels^2]"}
    maxdisp = max(resv[1]+resv[2], resv2[1]+resv2[2])
    if(is.na(maxdisp)) maxdisp=1.1*max(resv[1], resv2[1])
    plotresnumbers_min(resv[GroupOrder,],paste0(NamesOfChannels[ChannelNum + 1] ," Size"), lbl, maxdisp)
    plotresnumbers_min(resv2[GroupOrder,],paste0(NamesOfChannels[ChannelNumColoc + 1] ," Size"), lbl, maxdisp)
    
    getMultiValue[resv1, resv2]=calculateStat(uniqueNames, dataGroups, NumberImagesPerEachGroup, NumberOfGroups, ChannelNum, ChannelNumColoc, MinObjSize, MaxObjSize, VES_TOTAL_SIZE)   
    if (isData3D){lbl="Total volume [pixels^3]"}else{lbl="Total area [pixels^2]"}
    maxdisp = max(resv1[1]+resv1[2], resv2[1]+resv2[2])
    if(is.na(maxdisp)) maxdisp=1.1*max(resv1[1], resv2[1])
    plotresnumbers_min(resv1[GroupOrder,],paste0(NamesOfChannels[ChannelNum + 1] ," Total size"), lbl, maxdisp)
    plotresnumbers_min(resv2[GroupOrder,],paste0(NamesOfChannels[ChannelNumColoc + 1] ," Total size"), lbl, maxdisp)
    
    if (isData3D){lbl="Total volume ratio"}else{lbl="Total area ratio"}
    getMultiValue[resv3, resv4]=calculateStat(uniqueNames, dataGroups, NumberImagesPerEachGroup, NumberOfGroups, ChannelNum, ChannelNumColoc, MinObjSize, MaxObjSize, VES_SIZE_RATIO)
    plotresnumbers_min(resv3[GroupOrder,],paste0("Total size ratio ", NamesOfChannels[ChannelNum + 1], "/", NamesOfChannels[ChannelNumColoc + 1]), lbl, 1.25*max(resv3))
    plotresnumbers_min(resv4[GroupOrder,],paste0("Total size ratio ", NamesOfChannels[ChannelNumColoc + 1], "/", NamesOfChannels[ChannelNum + 1]), lbl, 1.25*max(resv4))
    
    if (isData3D){lbl="Volume and number of objects"}else{lbl="Area and number of objects"}
    mtext(paste0(lbl,", ", min(NumberImagesPerEachGroup), " samples per group" ), 3, line=0, adj=0.5, cex=1.2, outer=TRUE)
    mtext(as.character(scriptInputParamsStr), 1, line=3, adj=0, cex=0.55, outer=TRUE)
    mtext(format(Sys.time(), "%Y-%m-%d "), cex=0.75, line=3, side=SOUTH<-1, adj=1, outer=TRUE)
    
    dev.off()
    
    ### Generate Secondary_results_B ###################################################
    debugPrint(paste0("++++++++++ Secondary_results B +++++++++++++++++++ channels: ", channelsStr))
    colocObjectResults2PdfName=paste0(outputPath, "Secondary_resultsB_", channelsStr, ".pdf")
    debugPrint(paste0("Output PDF: ", colocObjectResults2PdfName))
    pdf(colocObjectResults2PdfName, width=8.2, height=11.3)
    par(mfrow=c(4,2), mar= c(4, 4, 1, 1) + 0.1, oma =c(4,0,1.5,0.2) )
    
    # Mean object lengths
    getMultiValue[resv1, resv2]=calculateStat(uniqueNames, dataGroups, NumberImagesPerEachGroup, NumberOfGroups, ChannelNum, ChannelNumColoc, MinObjSize, MaxObjSize, VES_LENGHT)   
    maxdisp = max(resv1[1]+resv1[2], resv2[1]+resv2[2])
    if(is.na(maxdisp)) maxdisp=1.1*max(resv1[1], resv2[1])
    plotresnumbers_min(resv1[GroupOrder,], paste0(NamesOfChannels[ChannelNum + 1], " Length"), "Length [pixels]", maxdisp)
    plotresnumbers_min(resv2[GroupOrder,], paste0(NamesOfChannels[ChannelNumColoc + 1], " Length"), "Length [pixels]", maxdisp)
    
    # Mean object intensities
    getMultiValue[resv1, resv2]=calculateStat(uniqueNames, dataGroups, NumberImagesPerEachGroup, NumberOfGroups, ChannelNum, ChannelNumColoc, MinObjSize, MaxObjSize, VES_INTENSITY)   
    maxdisp = max(resv1[1]+resv1[2], resv2[1]+resv2[2])
    if(is.na(maxdisp)) maxdisp=1.1*max(resv1[1], resv2[1])
    plotresnumbers_min(resv1[GroupOrder,], paste0(NamesOfChannels[ChannelNum + 1], " Intensity"), "Intensity", maxdisp)
    plotresnumbers_min(resv2[GroupOrder,], paste0(NamesOfChannels[ChannelNumColoc + 1], " Intensity"), "Intensity", maxdisp)
    
    # Size threshold effect ch1 in ch2
    generateThreshold(ChannelNum, ChannelNumColoc, ColocABname, SizeThresholdsRange, THR_SIZE, "Min Size")
    generateThreshold(ChannelNumColoc, ChannelNum, ColocBAname, SizeThresholdsRange, THR_SIZE, "Min Size")
    generateThreshold(ChannelNum, ChannelNumColoc, ColocBAname, IntensityThresholdRange, THR_INTENSITY, "Min Intensity")
    generateThreshold(ChannelNumColoc, ChannelNum, ColocBAname, IntensityThresholdRange, THR_INTENSITY, "Min Intensity")
    
    mtext(paste0("Objects lengths, intensities and threshold effects, ", min(NumberImagesPerEachGroup), " samples per group" ), 3, line=0, adj=0.5, cex=1.2, outer=TRUE)
    mtext(as.character(scriptInputParamsStr), 1, line=3, adj=0, cex=0.55, outer=TRUE)
    mtext(format(Sys.time(), "%Y-%m-%d "), cex=0.75, line=3, side=SOUTH<-1, adj=1, outer=TRUE)
    
    dev.off()
}
