/*
 * Decompiled with CFR 0.152.
 */
package qupath.lib.color;

import java.io.IOException;
import java.net.URI;
import java.nio.file.FileSystem;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.TreeMap;
import java.util.stream.DoubleStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import qupath.lib.common.ColorTools;
import qupath.lib.common.GeneralTools;

public class ColorMaps {
    private static final Logger logger = LoggerFactory.getLogger(ColorMaps.class);
    private static ColorMap LEGACY_COLOR_MAP = new PseudoColorMap();
    private static List<ColorMap> SINGLE_COLOR_MAPS = Arrays.asList(ColorMaps.createColorMap("Gray", 255, 255, 255), ColorMaps.createColorMap("Red", 255, 0, 0), ColorMaps.createColorMap("Green", 0, 255, 0), ColorMaps.createColorMap("Blue", 0, 0, 255), ColorMaps.createColorMap("Magenta", 255, 0, 255), ColorMaps.createColorMap("Yellow", 255, 255, 0), ColorMaps.createColorMap("Cyan", 0, 255, 255));
    private static ColorMap defaultColorMap = LEGACY_COLOR_MAP;
    private static List<String> preferredDefaultColorMaps = Arrays.asList("Inferno", "Plasma", "Magma", "Viridis");
    private static Map<String, ColorMap> maps = new LinkedHashMap<String, ColorMap>(ColorMaps.loadDefaultColorMaps());
    private static Map<String, ColorMap> mapsUnmodifiable = Collections.unmodifiableMap(maps);

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static Map<String, ColorMap> loadDefaultColorMaps() {
        LinkedHashMap<String, ColorMap> maps = new LinkedHashMap<String, ColorMap>();
        try {
            Iterator<String> pathColorMaps;
            URI uri = ColorMaps.class.getResource("/colormaps").toURI();
            if (uri.getScheme().equals("jar")) {
                FileSystem fileSystem = FileSystems.newFileSystem(uri, Map.of());
                pathColorMaps = fileSystem.getPath("/colormaps", new String[0]);
            } else {
                pathColorMaps = Paths.get(uri);
            }
            for (ColorMap cm : ColorMaps.loadColorMapsFromDirectory((Path)((Object)pathColorMaps))) {
                maps.put(cm.getName(), cm);
            }
        }
        catch (Exception e) {
            logger.error("Error loading default colormaps: " + e.getLocalizedMessage(), (Throwable)e);
            TreeMap<String, ColorMap> treeMap = new TreeMap<String, ColorMap>();
            return treeMap;
        }
        finally {
            for (ColorMap cm : SINGLE_COLOR_MAPS) {
                maps.put(cm.getName(), cm);
            }
            maps.put(LEGACY_COLOR_MAP.getName(), LEGACY_COLOR_MAP);
            for (String preferred : preferredDefaultColorMaps) {
                if (!maps.containsKey(preferred)) continue;
                defaultColorMap = (ColorMap)maps.get(preferred);
                break;
            }
        }
        for (ColorMap cm : SINGLE_COLOR_MAPS) {
            maps.put(cm.getName(), cm);
        }
        maps.put(LEGACY_COLOR_MAP.getName(), LEGACY_COLOR_MAP);
        for (String preferred : preferredDefaultColorMaps) {
            if (!maps.containsKey(preferred)) continue;
            defaultColorMap = (ColorMap)maps.get(preferred);
            break;
        }
        return maps;
    }

    public static boolean installColorMaps(Path ... paths) {
        boolean changes = false;
        for (Path p : paths) {
            try {
                ColorMap cm;
                if (Files.isDirectory(p, new LinkOption[0])) {
                    for (ColorMap cm2 : ColorMaps.loadColorMapsFromDirectory(p)) {
                        maps.put(cm2.getName(), cm2);
                        changes = true;
                    }
                    continue;
                }
                if (!Files.isRegularFile(p, new LinkOption[0]) || (cm = ColorMaps.loadColorMap(p)) == null) continue;
                maps.put(cm.getName(), cm);
                changes = true;
            }
            catch (Exception e) {
                logger.error("Error loading custom colormaps", (Throwable)e);
            }
        }
        return changes;
    }

    public static boolean installColorMaps(ColorMap ... colorMaps) {
        boolean changes = false;
        for (ColorMap cm : colorMaps) {
            maps.put(cm.getName(), cm);
            changes = true;
        }
        return changes;
    }

    public static int[] getColors(ColorMap map, int nValues, boolean doInvert) {
        int[] vals = new int[nValues];
        double max = nValues - 1;
        for (int i = 0; i < nValues; ++i) {
            double value = i;
            if (doInvert) {
                value = 1.0 - value;
            }
            vals[i] = map.getColor(value, 0.0, max);
        }
        return vals;
    }

    public static ColorMap getDefaultColorMap() {
        return defaultColorMap;
    }

    public static void setDefaultColorMap(ColorMap colorMap) {
        Objects.requireNonNull(colorMap);
        defaultColorMap = colorMap;
    }

    public static Map<String, ColorMap> getColorMaps() {
        return mapsUnmodifiable;
    }

    private static List<ColorMap> loadColorMapsFromDirectory(Path path) throws IOException {
        ArrayList<ColorMap> list = new ArrayList<ColorMap>();
        Iterator iter = Files.list(path).filter(p -> p.getFileName().toString().endsWith(".tsv")).sorted(Comparator.comparing(p -> p.getFileName().toString())).iterator();
        while (iter.hasNext()) {
            Path temp = (Path)iter.next();
            try {
                list.add(ColorMaps.loadColorMap(temp));
            }
            catch (Exception e) {
                logger.error("Error loading colormap from {}", (Object)temp);
            }
        }
        return list;
    }

    private static ColorMap loadColorMap(Path path) throws IOException {
        String name = path.getFileName().toString();
        if (name.endsWith(".tsv")) {
            name = name.substring(0, name.length() - 4);
        }
        List<String> lines = Files.readAllLines(path).stream().filter(s -> !s.isBlank()).toList();
        int n = lines.size();
        double[] r = new double[n];
        double[] g = new double[n];
        double[] b = new double[n];
        int i = 0;
        for (String line : lines) {
            String[] split = line.split("\\s+");
            if (split.length == 0) continue;
            if (split.length < 3) {
                logger.warn("Invalid line (must contain 3 doubles): {}", (Object)line);
                continue;
            }
            r[i] = Double.parseDouble(split[0]);
            g[i] = Double.parseDouble(split[1]);
            b[i] = Double.parseDouble(split[2]);
            ++i;
        }
        if (i < n) {
            r = Arrays.copyOf(r, i);
            g = Arrays.copyOf(g, i);
            b = Arrays.copyOf(b, i);
        }
        return ColorMaps.createColorMap(name, r, g, b);
    }

    public static ColorMap createColorMap(String name, double[] r, double[] g, double[] b) {
        int[] ri = ColorMaps.convertToInt(r);
        int[] gi = ColorMaps.convertToInt(g);
        int[] bi = ColorMaps.convertToInt(b);
        return ColorMaps.createColorMap(name, ri, gi, bi);
    }

    private static int[] convertToInt(double[] arr) {
        return DoubleStream.of(arr).mapToInt(d -> ColorMaps.convertToInt(d)).toArray();
    }

    private static int convertToInt(double d) {
        if (!Double.isFinite(d) || d > 1.0 || d < 0.0) {
            throw new IllegalArgumentException("Color value must be between 0 and 1, but actual value is " + d);
        }
        return (int)Math.round(d * 255.0);
    }

    public static ColorMap createColorMap(String name, int[] r, int[] g, int[] b) {
        return new DefaultColorMap(name, r, g, b);
    }

    public static ColorMap createColorMap(String name, int r, int g, int b) {
        return new SingleColorMap(name, 0, r, 0, g, 0, b);
    }

    public static ColorMap gammaColorMap(ColorMap map, double gamma) {
        return new GammaColorMap(map, gamma);
    }

    static int getInd(double value, double minValue, double maxValue, int nColors) {
        int ind = 0;
        double max = Math.max(minValue, maxValue);
        double min = Math.min(minValue, maxValue);
        ind = (int)Math.round((value - min) / (max - min) * (double)nColors);
        ind = ind >= nColors ? nColors - 1 : ind;
        ind = Math.max(ind, 0);
        return minValue > maxValue ? nColors - 1 - ind : ind;
    }

    public static interface ColorMap {
        public String getName();

        public Integer getColor(double var1, double var3, double var5);
    }

    private static class DefaultColorMap
    implements ColorMap {
        private String name;
        private final int[] r;
        private final int[] g;
        private final int[] b;
        private int nColors = 256;
        private Integer[] colors = new Integer[this.nColors];

        DefaultColorMap(String name, int[] r, int[] g, int[] b) {
            this.name = name;
            this.r = (int[])r.clone();
            this.g = (int[])g.clone();
            this.b = (int[])b.clone();
            double scale = (double)(r.length - 1) / (double)this.nColors;
            for (int i = 0; i < this.nColors; ++i) {
                int ind = (int)((double)i * scale);
                double residual = (double)i * scale - (double)ind;
                this.colors[i] = ColorTools.packRGB(r[ind] + (int)((double)(r[ind + 1] - r[ind]) * residual), g[ind] + (int)((double)(g[ind + 1] - g[ind]) * residual), b[ind] + (int)((double)(b[ind + 1] - b[ind]) * residual));
            }
            this.colors[this.nColors - 1] = ColorTools.packRGB(r[r.length - 1], g[g.length - 1], b[b.length - 1]);
        }

        @Override
        public String getName() {
            return this.name;
        }

        public String toString() {
            return this.getName();
        }

        private Integer getColor(int ind) {
            Integer color = this.colors[ind];
            if (color == null) {
                this.colors[ind] = color = Integer.valueOf(ColorTools.packRGB(this.r[ind], this.g[ind], this.b[ind]));
            }
            return color;
        }

        @Override
        public Integer getColor(double value, double minValue, double maxValue) {
            return this.getColor(ColorMaps.getInd(value, minValue, maxValue, this.nColors));
        }
    }

    private static class SingleColorMap
    implements ColorMap {
        private String name;
        private final int minRed;
        private final int maxRed;
        private final int minGreen;
        private final int maxGreen;
        private final int minBlue;
        private final int maxBlue;

        SingleColorMap(String name, int minRed, int maxRed, int minGreen, int maxGreen, int minBlue, int maxBlue) {
            this.name = name;
            this.minRed = minRed;
            this.maxRed = maxRed;
            this.minGreen = minGreen;
            this.maxGreen = maxGreen;
            this.minBlue = minBlue;
            this.maxBlue = maxBlue;
        }

        @Override
        public String getName() {
            return this.name;
        }

        public String toString() {
            return this.getName();
        }

        @Override
        public Integer getColor(double value, double minValue, double maxValue) {
            if (minValue == maxValue) {
                return ColorTools.packRGB(this.maxRed, this.maxGreen, this.maxBlue);
            }
            double val = (value - minValue) / (maxValue - minValue);
            if (val >= 1.0) {
                return ColorTools.packRGB(this.maxRed, this.maxGreen, this.maxBlue);
            }
            if (val <= 0.0) {
                return ColorTools.packRGB(this.minRed, this.minGreen, this.minBlue);
            }
            int r = (int)Math.round(SingleColorMap.linearInterp1D(this.minRed, this.maxRed, val));
            int g = (int)Math.round(SingleColorMap.linearInterp1D(this.minGreen, this.maxGreen, val));
            int b = (int)Math.round(SingleColorMap.linearInterp1D(this.minBlue, this.maxBlue, val));
            return ColorTools.packRGB(r, g, b);
        }

        private static double linearInterp1D(double fx0, double fx1, double x) {
            assert (x >= 0.0 && x <= 1.0);
            return fx0 * (1.0 - x) + x * fx1;
        }
    }

    private static class GammaColorMap
    implements ColorMap {
        private final ColorMap map;
        private final double gamma;

        public GammaColorMap(ColorMap map, double gamma) {
            this.map = map;
            this.gamma = gamma;
        }

        @Override
        public String getName() {
            return String.format("%s (gamma=%s)", this.map.getName(), GeneralTools.formatNumber(this.gamma, 3));
        }

        @Override
        public Integer getColor(double value, double minValue, double maxValue) {
            if (this.gamma == 1.0) {
                return this.map.getColor(value, minValue, maxValue);
            }
            double val = 0.0;
            if (this.gamma == 0.0) {
                val = 1.0;
            } else if (maxValue != minValue) {
                val = (value - minValue) / (maxValue - minValue);
            }
            val = Math.pow(val, this.gamma);
            return this.map.getColor(val, 0.0, 1.0);
        }
    }

    private static class PseudoColorMap
    implements ColorMap {
        private static final int[] r = new int[]{0, 0, 0, 0, 255, 255};
        private static final int[] g = new int[]{0, 0, 255, 255, 255, 0};
        private static final int[] b = new int[]{0, 255, 255, 0, 0, 0};
        private static final int nColors = 256;
        private static final Integer[] colors = new Integer[256];

        private PseudoColorMap() {
        }

        public String toString() {
            return this.getName() + " (legacy)";
        }

        @Override
        public String getName() {
            return "Jet";
        }

        private Integer getColor(int ind) {
            Integer color = colors[ind];
            if (color == null) {
                PseudoColorMap.colors[ind] = color = Integer.valueOf(ColorTools.packRGB(r[ind], g[ind], b[ind]));
            }
            return color;
        }

        @Override
        public Integer getColor(double value, double minValue, double maxValue) {
            return this.getColor(ColorMaps.getInd(value, minValue, maxValue, 256));
        }

        static {
            double scale = (double)(r.length - 1) / 256.0;
            for (int i = 0; i < 256; ++i) {
                int ind = (int)((double)i * scale);
                double residual = (double)i * scale - (double)ind;
                PseudoColorMap.colors[i] = ColorTools.packRGB(r[ind] + (int)((double)(r[ind + 1] - r[ind]) * residual), g[ind] + (int)((double)(g[ind + 1] - g[ind]) * residual), b[ind] + (int)((double)(b[ind + 1] - b[ind]) * residual));
            }
            PseudoColorMap.colors[255] = ColorTools.packRGB(r[r.length - 1], g[g.length - 1], b[b.length - 1]);
        }
    }
}

