/*
 * Decompiled with CFR 0.152.
 */
package qupath.lib.images.servers;

import java.awt.Graphics2D;
import java.awt.RenderingHints;
import java.awt.image.BufferedImage;
import java.awt.image.ColorModel;
import java.awt.image.WritableRaster;
import java.io.IOException;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.FutureTask;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import qupath.lib.awt.common.BufferedImageTools;
import qupath.lib.color.ColorModelFactory;
import qupath.lib.images.servers.AbstractImageServer;
import qupath.lib.images.servers.ImageServerMetadata;
import qupath.lib.images.servers.TileRequest;
import qupath.lib.regions.RegionRequest;

public abstract class AbstractTileableImageServer
extends AbstractImageServer<BufferedImage> {
    private static final Logger logger = LoggerFactory.getLogger(AbstractTileableImageServer.class);
    private ColorModel colorModel;
    private Map<String, BufferedImage> emptyTileMap = new HashMap<String, BufferedImage>();
    private transient Set<TileRequest> emptyTiles = new HashSet<TileRequest>();
    private static final Long ZERO = 0L;
    private transient Set<RegionRequest> failedCacheTiles = new HashSet<RegionRequest>();
    private Map<TileRequest, TileTask> pendingTiles = new ConcurrentHashMap<TileRequest, TileTask>();
    private int duplicateRequestClashCount = 0;

    protected AbstractTileableImageServer() {
        super(BufferedImage.class);
    }

    protected BufferedImage getEmptyTile(int width, int height) throws IOException {
        return this.getEmptyTile(width, height, true);
    }

    protected BufferedImage getEmptyTile(int width, int height, boolean doCache) throws IOException {
        String key = width + "x" + height;
        BufferedImage imgTile = this.emptyTileMap.get(key);
        if (imgTile == null) {
            imgTile = AbstractTileableImageServer.createEmptyTile(this.getDefaultColorModel(), width, height);
            if (doCache) {
                this.emptyTileMap.put(key, imgTile);
            }
        }
        return imgTile;
    }

    protected ColorModel getDefaultColorModel() throws IOException {
        if (this.colorModel == null) {
            this.colorModel = this.isRGB() ? ColorModel.getRGBdefault() : ColorModelFactory.createColorModel(this.getPixelType(), this.getMetadata().getChannels());
        }
        return this.colorModel;
    }

    static BufferedImage createEmptyTile(ColorModel colorModel, int width, int height) {
        WritableRaster raster = colorModel.createCompatibleWritableRaster(width, height);
        Hashtable<String, Long> emptyProperties = new Hashtable<String, Long>();
        emptyProperties.put("QUPATH_CACHE_SIZE", ZERO);
        return new BufferedImage(colorModel, raster, false, emptyProperties);
    }

    static boolean isEmptyTile(BufferedImage img) {
        return ZERO.equals(img.getProperty("QUPATH_CACHE_SIZE"));
    }

    synchronized void resetEmptyTileCache() {
        logger.debug("Resetting empty tile cache");
        this.emptyTileMap.clear();
    }

    protected abstract BufferedImage readTile(TileRequest var1) throws IOException;

    protected BufferedImage getTile(TileRequest tileRequest) throws IOException {
        BufferedImage imgCached;
        RegionRequest request = tileRequest.getRegionRequest();
        if (this.emptyTiles.contains(tileRequest)) {
            return this.getEmptyTile(tileRequest.getTileWidth(), tileRequest.getTileHeight());
        }
        Map cache = this.getCache();
        if (cache != null && (imgCached = (BufferedImage)cache.get(request)) != null) {
            logger.trace("Returning cached tile: {}", (Object)request);
            return imgCached;
        }
        logger.trace("Reading tile: {}", (Object)request);
        TileTask futureTask = this.pendingTiles.computeIfAbsent(tileRequest, t -> new TileTask(Thread.currentThread(), () -> this.readTile((TileRequest)t)));
        boolean myTask = futureTask.thread == Thread.currentThread();
        try {
            if (myTask) {
                futureTask.run();
            } else {
                ++this.duplicateRequestClashCount;
                logger.debug("Duplicate request for a pending tile ({} total) - {}", (Object)this.duplicateRequestClashCount, (Object)tileRequest.getRegionRequest());
            }
            imgCached = (BufferedImage)futureTask.get();
        }
        catch (InterruptedException | ExecutionException e) {
            if (e.getCause() instanceof IOException) {
                throw (IOException)e.getCause();
            }
            throw new IOException(e);
        }
        if (myTask) {
            if (imgCached != null) {
                if (AbstractTileableImageServer.isEmptyTile(imgCached)) {
                    this.emptyTiles.add(tileRequest);
                } else if (cache != null) {
                    cache.put(request, imgCached);
                    if (!cache.containsKey(request) && this.failedCacheTiles.add(request)) {
                        logger.warn("Unable to add {} to cache.\nYou might need to give QuPath more memory, or to increase the 'Percentage memory for tile caching' preference.", (Object)request);
                    }
                }
            }
            this.pendingTiles.remove(tileRequest);
        }
        return imgCached;
    }

    protected BufferedImage createDefaultRGBImage(int width, int height) {
        return new BufferedImage(width, height, 1);
    }

    @Override
    public BufferedImage readRegion(RegionRequest request) throws IOException {
        TileRequest firstTile;
        boolean singleTile;
        BufferedImage img;
        Map cache = this.getCache();
        BufferedImage bufferedImage = img = request.getPath().equals(this.getPath()) && cache != null ? (BufferedImage)cache.get(request) : null;
        if (img != null) {
            return BufferedImageTools.duplicate(img);
        }
        Collection<TileRequest> tiles = this.getTileRequestManager().getTileRequests(request);
        if (tiles.isEmpty()) {
            return null;
        }
        boolean bl = singleTile = tiles.size() == 1;
        if (singleTile && (firstTile = tiles.iterator().next()).getRegionRequest().equals(request)) {
            BufferedImage imgTile = this.getTile(firstTile);
            if (imgTile == null) {
                return null;
            }
            return BufferedImageTools.duplicate(imgTile);
        }
        this.prerequestTiles(tiles);
        int width = (int)Math.max(1L, Math.round((double)request.getWidth() / request.getDownsample()));
        int height = (int)Math.max(1L, Math.round((double)request.getHeight() / request.getDownsample()));
        long startTime = System.currentTimeMillis();
        if (this.isRGB()) {
            BufferedImage imgResult = this.createRGBImage(request, tiles, width, height);
            Graphics2D g2d = imgResult.createGraphics();
            g2d.scale(1.0 / request.getDownsample(), 1.0 / request.getDownsample());
            g2d.translate(-request.getX(), -request.getY());
            if (request.getDownsample() > 1.0) {
                g2d.setRenderingHint(RenderingHints.KEY_INTERPOLATION, RenderingHints.VALUE_INTERPOLATION_BILINEAR);
            }
            for (TileRequest tileRequest : tiles) {
                BufferedImage imgTile = this.getTile(tileRequest);
                g2d.drawImage(imgTile, tileRequest.getImageX(), tileRequest.getImageY(), tileRequest.getImageWidth(), tileRequest.getImageHeight(), null);
            }
            g2d.dispose();
            long endTime = System.currentTimeMillis();
            logger.trace("Requested " + tiles.size() + " tiles in " + (endTime - startTime) + " ms (RGB)");
            return this.resizeIfNeeded(imgResult, width, height);
        }
        WritableRaster raster = null;
        ColorModel colorModel = null;
        boolean alphaPremultiplied = false;
        int tileMinX = Integer.MAX_VALUE;
        int tileMinY = Integer.MAX_VALUE;
        int tileMaxX = Integer.MIN_VALUE;
        int tileMaxY = Integer.MIN_VALUE;
        double tileDownsample = Double.NaN;
        for (TileRequest tileRequest : tiles) {
            if (Double.isNaN(tileDownsample)) {
                tileDownsample = tileRequest.getRegionRequest().getDownsample();
            }
            tileMinX = Math.min(tileRequest.getTileX(), tileMinX);
            tileMinY = Math.min(tileRequest.getTileY(), tileMinY);
            tileMaxX = Math.max(tileRequest.getTileX() + tileRequest.getTileWidth(), tileMaxX);
            tileMaxY = Math.max(tileRequest.getTileY() + tileRequest.getTileHeight(), tileMaxY);
        }
        boolean isEmptyRegion = true;
        if (singleTile) {
            BufferedImage imgTile = this.getTile(tiles.iterator().next());
            if (imgTile != null) {
                raster = imgTile.getRaster();
                colorModel = imgTile.getColorModel();
                alphaPremultiplied = imgTile.isAlphaPremultiplied();
                isEmptyRegion = AbstractTileableImageServer.isEmptyTile(imgTile);
            }
        } else {
            for (TileRequest tileRequest : tiles) {
                BufferedImage imgTile = this.getTile(tileRequest);
                if (imgTile == null || AbstractTileableImageServer.isEmptyTile(imgTile)) continue;
                isEmptyRegion = false;
                int dx = tileRequest.getTileX() - tileMinX;
                int dy = tileRequest.getTileY() - tileMinY;
                int tileWidth = tileMaxX - tileMinX;
                int tileHeight = tileMaxY - tileMinY;
                if (raster == null) {
                    raster = imgTile.getRaster().createCompatibleWritableRaster(tileWidth, tileHeight);
                    colorModel = imgTile.getColorModel();
                    alphaPremultiplied = imgTile.isAlphaPremultiplied();
                }
                if (dx >= raster.getWidth() || dy >= raster.getHeight()) continue;
                AbstractTileableImageServer.copyPixels(imgTile.getRaster(), dx, dy, raster);
            }
        }
        if (isEmptyRegion) {
            return this.getEmptyTile(request.getWidth(), request.getHeight());
        }
        if (raster == null) {
            return null;
        }
        int xStart = (int)Math.round((double)request.getX() / tileDownsample) - tileMinX;
        int yStart = (int)Math.round((double)request.getY() / tileDownsample) - tileMinY;
        int xEnd = (int)Math.round((double)(request.getX() + request.getWidth()) / tileDownsample) - tileMinX;
        int yEnd = (int)Math.round((double)(request.getY() + request.getHeight()) / tileDownsample) - tileMinY;
        if (xEnd > this.getWidth() || yEnd > this.getHeight()) {
            logger.warn("Region request is too large for {}x{} image: {}", new Object[]{this.getWidth(), this.getHeight(), request});
        } else if (xEnd - xStart <= 0 || yEnd - yStart <= 0) {
            return null;
        }
        if (xStart > 0 || yStart > 0 || xEnd != raster.getWidth() || yEnd != raster.getHeight()) {
            int x = Math.max(xStart, 0);
            int y = Math.max(yStart, 0);
            int w = xEnd - xStart;
            int h = yEnd - yStart;
            WritableRaster raster2 = raster.createCompatibleWritableRaster(w, h);
            AbstractTileableImageServer.copyPixels(raster, -x, -y, raster2);
            raster = raster2;
        }
        BufferedImage imgResult = new BufferedImage(colorModel, raster, alphaPremultiplied, null);
        imgResult = this.resizeIfNeeded(imgResult, width, height);
        long endTime = System.currentTimeMillis();
        logger.trace("Requested " + tiles.size() + " tiles in " + (endTime - startTime) + " ms (non-RGB)");
        return imgResult;
    }

    private BufferedImage resizeIfNeeded(BufferedImage img, int width, int height) {
        int currentWidth = img.getWidth();
        int currentHeight = img.getHeight();
        if (currentWidth != width || currentHeight != height) {
            logger.debug("Region size updated from {}x{} to {}x{}", new Object[]{currentWidth, currentHeight, width, height});
            return BufferedImageTools.resize(img, width, height, this.allowSmoothInterpolation());
        }
        return img;
    }

    private BufferedImage createRGBImage(RegionRequest request, Collection<TileRequest> tiles, int expectedWidth, int expectedHeight) {
        int imgWidth = expectedWidth;
        int imgHeight = expectedHeight;
        if (request.getDownsample() > 1.0 && this.nResolutions() > 1 && (request.getMaxX() == this.getWidth() || request.getMaxY() == this.getHeight()) && request.getMinX() >= 0 && request.getMinY() >= 0) {
            int maxX = -2147483647;
            int maxY = -2147483647;
            for (TileRequest tile : tiles) {
                maxX = Math.max(maxX, tile.getRegionRequest().getMaxX());
                maxY = Math.max(maxY, tile.getRegionRequest().getMaxY());
            }
            if (maxX < request.getMaxX() || maxY < request.getMaxY()) {
                int width2 = (int)Math.max(1L, Math.round((double)(maxX - request.getMinX()) / request.getDownsample() - 1.0E-9));
                int height2 = (int)Math.max(1L, Math.round((double)(maxY - request.getMinY()) / request.getDownsample() - 1.0E-9));
                if (expectedWidth == width2 + 1 || expectedHeight == height2 + 1) {
                    logger.trace("RGB image size updated from {}x{} to {}x{} to avoid border problems", new Object[]{expectedWidth, expectedHeight, width2, height2});
                    imgWidth = width2;
                    imgHeight = height2;
                }
            }
        }
        return this.createDefaultRGBImage(imgWidth, imgHeight);
    }

    private void prerequestTiles(Collection<TileRequest> tiles) {
        Map cache = this.getCache();
        for (TileRequest tile : tiles) {
            if (cache != null && (cache.containsKey(tile.getRegionRequest()) || this.pendingTiles.containsKey(tile))) continue;
            TileTask futureTask = this.pendingTiles.computeIfAbsent(tile, t -> new TileTask(Thread.currentThread(), () -> this.readTile((TileRequest)t)));
            if (futureTask.thread != Thread.currentThread()) continue;
            futureTask.run();
        }
    }

    private static void copyPixels(WritableRaster source, int dx, int dy, WritableRaster dest) {
        if (dest.getClass().getName().contains("ByteInterleavedRaster") && (dx < 0 || dy < 0)) {
            int sx = Math.max(-dx, 0);
            int sy = Math.max(-dy, 0);
            if (sx >= source.getWidth() || sy >= source.getHeight()) {
                logger.warn("No pixels needed to be copied!");
                return;
            }
            dest.setRect(0, 0, source.createChild(sx, sy, source.getWidth() - sx, source.getHeight() - sy, 0, 0, null));
        } else {
            dest.setRect(dx, dy, source);
        }
    }

    protected boolean allowSmoothInterpolation() {
        return this.getMetadata().getChannelType() != ImageServerMetadata.ChannelType.CLASSIFICATION;
    }

    private static class TileTask
    extends FutureTask<BufferedImage> {
        private Thread thread;

        public TileTask(Thread thread, Callable<BufferedImage> callable) {
            super(callable);
            this.thread = thread;
        }
    }
}

