/*
 * Decompiled with CFR 0.152.
 */
package qupath.lib.images.writers;

import com.google.gson.Gson;
import java.awt.image.BufferedImage;
import java.awt.image.IndexColorModel;
import java.awt.image.WritableRaster;
import java.io.BufferedWriter;
import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.text.NumberFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import qupath.lib.awt.common.BufferedImageTools;
import qupath.lib.common.GeneralTools;
import qupath.lib.common.ThreadTools;
import qupath.lib.images.ImageData;
import qupath.lib.images.servers.ImageServer;
import qupath.lib.images.servers.ImageServerMetadata;
import qupath.lib.images.servers.LabeledImageServer;
import qupath.lib.images.servers.TransformedServerBuilder;
import qupath.lib.images.writers.ImageWriterTools;
import qupath.lib.io.GsonTools;
import qupath.lib.objects.PathObject;
import qupath.lib.objects.PathObjectTools;
import qupath.lib.objects.classes.PathClass;
import qupath.lib.regions.ImageRegion;
import qupath.lib.regions.Padding;
import qupath.lib.regions.RegionRequest;
import qupath.lib.roi.RoiTools;
import qupath.lib.roi.interfaces.ROI;

public class TileExporter {
    private static final Logger logger = LoggerFactory.getLogger(TileExporter.class);
    private ImageData<BufferedImage> imageData;
    private ImageServer<BufferedImage> server;
    private ImageRegion region = null;
    private List<PathObject> parentObjects = null;
    private boolean useParentRoiBounds = false;
    private boolean preferNucleus = true;
    private double downsample;
    private int tileWidth = 512;
    private int tileHeight = 512;
    private int overlapX = 0;
    private int overlapY = 0;
    private boolean includePartialTiles = false;
    private boolean annotatedTilesOnly = false;
    private boolean annotatedCentroidTilesOnly = false;
    private int minZ = 0;
    private int minT = 0;
    private int maxZ = -1;
    private int maxT = -1;
    private String ext = ".tif";
    private String extLabeled = null;
    private String imageSubDir = null;
    private String labelSubDir = null;
    private boolean exportJson = false;
    private String labelId = null;
    private ImageServer<BufferedImage> serverLabeled;
    private static ThreadLocal<NumberFormat> formatter = ThreadLocal.withInitial(() -> TileExporter.createDefaultNumberFormat(5));

    public TileExporter(ImageData<BufferedImage> imageData) {
        this.imageData = imageData;
        this.server = imageData.getServer();
    }

    public TileExporter parentObjects(Predicate<PathObject> filter) {
        this.parentObjects = this.imageData.getHierarchy().getFlattenedObjectList(null).stream().filter(filter).toList();
        return this;
    }

    public TileExporter parentObjects(Collection<? extends PathObject> parentObjects) {
        this.parentObjects = new ArrayList<PathObject>(parentObjects);
        return this;
    }

    public TileExporter useROIBounds(boolean fullROIs) {
        this.useParentRoiBounds = fullROIs;
        return this;
    }

    public TileExporter fullImageTile() {
        this.parentObjects = Collections.singletonList(this.imageData.getHierarchy().getRootObject());
        return this;
    }

    public TileExporter tileSize(int tileSize) {
        return this.tileSize(tileSize, tileSize);
    }

    public TileExporter tileSize(int tileWidth, int tileHeight) {
        this.tileWidth = tileWidth;
        this.tileHeight = tileHeight;
        return this;
    }

    public TileExporter channels(int ... channels) {
        this.server = new TransformedServerBuilder(this.server).extractChannels(channels).build();
        return this;
    }

    public TileExporter channels(String ... channelNames) {
        this.server = new TransformedServerBuilder(this.server).extractChannels(channelNames).build();
        return this;
    }

    public TileExporter overlap(int overlap) {
        return this.overlap(overlap, overlap);
    }

    public TileExporter overlap(int overlapX, int overlapY) {
        this.overlapX = overlapX;
        this.overlapY = overlapY;
        return this;
    }

    public TileExporter downsample(double downsample) {
        this.downsample = downsample;
        return this;
    }

    public TileExporter requestedPixelSize(double pixelSize) {
        this.downsample = pixelSize / this.server.getPixelCalibration().getAveragedPixelSize().doubleValue();
        return this;
    }

    public TileExporter includePartialTiles(boolean includePartialTiles) {
        this.includePartialTiles = includePartialTiles;
        return this;
    }

    public TileExporter region(RegionRequest region) {
        this.region = region;
        this.downsample = region.getDownsample();
        return this;
    }

    public TileExporter region(ImageRegion region) {
        this.region = region;
        return this;
    }

    public TileExporter zRange(int minZ, int maxZ) {
        this.minZ = minZ;
        this.maxZ = maxZ;
        return this;
    }

    public TileExporter tRange(int minT, int maxT) {
        this.minT = minT;
        this.maxT = maxT;
        return this;
    }

    public TileExporter annotatedTilesOnly(boolean annotatedTilesOnly) {
        this.annotatedTilesOnly = annotatedTilesOnly;
        return this;
    }

    public TileExporter annotatedCentroidTilesOnly(boolean annotatedCentroidTilesOnly) {
        this.annotatedCentroidTilesOnly = annotatedCentroidTilesOnly;
        return this;
    }

    public TileExporter imageExtension(String ext) {
        if (!((String)ext).startsWith(".")) {
            ext = "." + (String)ext;
        }
        this.ext = ext;
        return this;
    }

    public TileExporter labeledImageExtension(String ext) {
        if (!((String)ext).startsWith(".")) {
            ext = "." + (String)ext;
        }
        this.extLabeled = ext;
        return this;
    }

    public TileExporter labeledServer(ImageServer<BufferedImage> server) {
        this.serverLabeled = server;
        return this;
    }

    public TileExporter imageSubDir(String subdir) {
        this.imageSubDir = subdir;
        return this;
    }

    public TileExporter labeledImageSubDir(String subdir) {
        this.labelSubDir = subdir;
        return this;
    }

    public TileExporter labeledImageId(String labelId) {
        this.labelId = labelId;
        return this;
    }

    public TileExporter exportJson(boolean exportJson) {
        this.exportJson = exportJson;
        return this;
    }

    private Collection<RegionRequestWrapper> createRequests() {
        ArrayList<RegionRequestWrapper> requests = new ArrayList<RegionRequestWrapper>();
        double downsample = this.downsample;
        if (downsample <= 0.0) {
            downsample = this.server.getDownsampleForResolution(0);
            if (this.downsample < 0.0) {
                logger.warn("Invalid downsample {}, I will use the level 0 downsample {}", (Object)this.downsample, (Object)downsample);
            } else {
                logger.debug("Using level 0 downsample {}", (Object)downsample);
            }
        }
        if (this.parentObjects == null) {
            requests.addAll(this.getTiledRegionRequests(downsample));
        } else {
            for (PathObject parent : this.parentObjects) {
                RegionRequest newRequest;
                int w = (int)Math.round((double)this.tileWidth * downsample);
                int h = (int)Math.round((double)this.tileHeight * downsample);
                if (parent.isRootObject()) {
                    for (int t = 0; t < this.server.nTimepoints(); ++t) {
                        for (int z = 0; z < this.server.nZSlices(); ++z) {
                            RegionRequest newRequest2;
                            if (this.useParentRoiBounds) {
                                newRequest2 = RegionRequest.createInstance(this.server.getPath(), downsample, 0, 0, this.server.getWidth(), this.server.getHeight(), z, t);
                            } else {
                                int x = (int)Math.round((double)this.server.getWidth() / 2.0 - (double)w / 2.0);
                                int y = (int)Math.round((double)this.server.getHeight() / 2.0 - (double)h / 2.0);
                                newRequest2 = RegionRequest.createInstance(this.server.getPath(), downsample, x, y, w, h, z, t);
                            }
                            if (!this.includePartialTiles && !TileExporter.withinImage(newRequest2, this.server)) continue;
                            requests.add(new RegionRequestWrapper(newRequest2, false));
                        }
                    }
                    continue;
                }
                if (!parent.hasROI()) continue;
                ROI roi = PathObjectTools.getROI(parent, this.preferNucleus);
                if (this.useParentRoiBounds) {
                    newRequest = RegionRequest.createInstance(this.server.getPath(), downsample, roi);
                } else {
                    int x = (int)Math.round(roi.getCentroidX() - (double)w / 2.0);
                    int y = (int)Math.round(roi.getCentroidY() - (double)h / 2.0);
                    newRequest = RegionRequest.createInstance(this.server.getPath(), downsample, x, y, w, h, roi.getImagePlane());
                }
                if (!this.includePartialTiles && !TileExporter.withinImage(newRequest, this.server)) continue;
                requests.add(new RegionRequestWrapper(newRequest, false));
            }
        }
        Iterator iterator = requests.iterator();
        while (iterator.hasNext()) {
            RegionRequest r = ((RegionRequestWrapper)iterator.next()).request;
            if (this.annotatedCentroidTilesOnly) {
                double cx = (double)(r.getMinX() + r.getMaxX()) / 2.0;
                double cy = (double)(r.getMinY() + r.getMaxY()) / 2.0;
                if (this.serverLabeled != null && this.serverLabeled instanceof LabeledImageServer) {
                    if (((LabeledImageServer)this.serverLabeled).getObjectsForRegion(r).stream().anyMatch(p -> p.getROI().contains(cx, cy))) continue;
                    logger.trace("Skipping empty labelled region based on centroid test {}", (Object)r);
                    iterator.remove();
                    continue;
                }
                if (this.imageData == null || !PathObjectTools.getObjectsForLocation(this.imageData.getHierarchy(), cx, cy, r.getZ(), r.getT(), 0.0).isEmpty()) continue;
                iterator.remove();
                continue;
            }
            if (!this.annotatedTilesOnly) continue;
            if (this.serverLabeled != null) {
                if (!this.serverLabeled.isEmptyRegion(r)) continue;
                logger.trace("Skipping empty labelled region {}", (Object)r);
                iterator.remove();
                continue;
            }
            if (this.imageData == null || this.imageData.getHierarchy().getAnnotationsForRegion(r, null).stream().anyMatch(p -> RoiTools.intersectsRegion(p.getROI(), r))) continue;
            iterator.remove();
        }
        return requests;
    }

    private static boolean withinImage(ImageRegion region, ImageServer<?> server) {
        return region.getX() >= 0 && region.getY() >= 0 && region.getMaxX() <= server.getWidth() && region.getMaxY() <= server.getHeight();
    }

    public void writeTiles(String dirOutput) throws IOException {
        Collection<RegionRequestWrapper> requests;
        if (!new File(dirOutput).isDirectory()) {
            throw new IOException("Output directory " + dirOutput + " does not exist!");
        }
        if (this.imageSubDir != null) {
            new File(dirOutput, this.imageSubDir).mkdirs();
        }
        if (this.labelSubDir != null) {
            new File(dirOutput, this.labelSubDir).mkdirs();
        }
        if (this.serverLabeled != null && this.extLabeled == null) {
            String string = this.extLabeled = this.serverLabeled.getMetadata().getChannelType() == ImageServerMetadata.ChannelType.CLASSIFICATION ? ".png" : ".tif";
        }
        if ((requests = this.createRequests()).isEmpty()) {
            logger.warn("No regions to export!");
            return;
        }
        if (requests.size() > 1) {
            logger.info("Exporting {} tiles", (Object)requests.size());
        }
        ExecutorService pool = Executors.newFixedThreadPool(ThreadTools.getParallelism(), ThreadTools.createThreadFactory("tile-exporter", true));
        String imageName = GeneralTools.stripInvalidFilenameChars(GeneralTools.stripExtension(this.server.getMetadata().getName()));
        Object imagePathName = null;
        Collection<URI> uris = this.server.getURIs();
        imagePathName = uris.isEmpty() ? imageName : (uris.size() == 1 ? uris.iterator().next().toString() : "[" + uris.stream().map(u -> u.toString()).collect(Collectors.joining("|")) + "]");
        int tileWidth = this.tileWidth;
        int tileHeight = this.tileHeight;
        ArrayList<TileExportEntry> exportImages = new ArrayList<TileExportEntry>();
        for (RegionRequestWrapper r : requests) {
            boolean ensureSize = !r.partialTile;
            String baseName = String.format("%s [%s]", imageName, TileExporter.getRegionString(r.request));
            Object exportImageName = baseName + this.ext;
            if (this.imageSubDir != null) {
                exportImageName = Paths.get(this.imageSubDir, new String[]{exportImageName}).toString();
            }
            String pathImageOutput = Paths.get(dirOutput, new String[]{exportImageName}).toAbsolutePath().toString();
            ExportTask taskImage = new ExportTask(this.server, r.request, pathImageOutput, tileWidth, tileHeight, ensureSize);
            Object exportLabelName = null;
            ExportTask taskLabels = null;
            if (this.serverLabeled != null) {
                Object labelName = baseName;
                if ((this.labelSubDir == null || this.labelSubDir.equals(this.imageSubDir)) && this.labelId == null && this.ext.equals(this.extLabeled)) {
                    labelName = baseName + "-labelled";
                } else if (this.labelId != null) {
                    labelName = baseName + this.labelId;
                }
                exportLabelName = (String)labelName + this.extLabeled;
                if (this.labelSubDir != null) {
                    exportLabelName = Paths.get(this.labelSubDir, new String[]{exportLabelName}).toString();
                }
                String pathLabelsOutput = Paths.get(dirOutput, new String[]{exportLabelName}).toAbsolutePath().toString();
                taskLabels = new ExportTask(this.serverLabeled, r.request.updatePath(this.serverLabeled.getPath()), pathLabelsOutput, tileWidth, tileHeight, ensureSize);
            }
            exportImages.add(new TileExportEntry(r.request.updatePath((String)imagePathName), (String)exportImageName, (String)exportLabelName));
            if (taskImage != null) {
                pool.submit(taskImage);
            }
            if (taskLabels == null) continue;
            pool.submit(taskLabels);
        }
        if (this.exportJson) {
            Path pathJson;
            Gson gson = GsonTools.getInstance(true).newBuilder().disableHtmlEscaping().create();
            TileExportData data = new TileExportData(dirOutput, exportImages);
            if (this.serverLabeled instanceof LabeledImageServer) {
                TileExportLabel label;
                PathClass pathClass;
                Map<PathClass, Integer> labels = ((LabeledImageServer)this.serverLabeled).getLabels();
                Map<PathClass, Integer> boundaryLabels = ((LabeledImageServer)this.serverLabeled).getBoundaryLabels();
                ArrayList<TileExportLabel> labelList = new ArrayList<TileExportLabel>();
                HashSet existingLabels = new HashSet();
                for (Map.Entry<PathClass, Integer> entry : labels.entrySet()) {
                    pathClass = entry.getKey();
                    label = new TileExportLabel(pathClass.toString(), entry.getValue(), boundaryLabels.getOrDefault(pathClass, null));
                    labelList.add(label);
                }
                for (Map.Entry<PathClass, Integer> entry : boundaryLabels.entrySet()) {
                    pathClass = entry.getKey();
                    if (existingLabels.contains(pathClass)) continue;
                    label = new TileExportLabel(pathClass.toString(), null, boundaryLabels.getOrDefault(pathClass, null));
                    labelList.add(label);
                }
                data.labels = labelList;
            }
            if (Files.exists(pathJson = Paths.get(dirOutput, imageName + "-tiles.json"), new LinkOption[0])) {
                logger.warn("Overwriting existing JSON file {}", (Object)pathJson);
            }
            try (BufferedWriter writer = Files.newBufferedWriter(pathJson, StandardCharsets.UTF_8, new OpenOption[0]);){
                gson.toJson((Object)data, (Appendable)writer);
            }
        }
        pool.shutdown();
        try {
            pool.awaitTermination(24L, TimeUnit.HOURS);
        }
        catch (InterruptedException e) {
            pool.shutdownNow();
            logger.error("Tile export interrupted: {}", (Object)e.getLocalizedMessage());
            logger.error("", (Throwable)e);
            throw new IOException(e);
        }
    }

    private static BufferedImage cropOrPad(BufferedImage img, int width, int height, double xProp, double yProp) {
        if (img.getWidth() != width || img.getHeight() != height) {
            if (img.getWidth() > width) {
                img = img.getHeight() > height ? BufferedImageTools.crop(img, 0, 0, width, height) : BufferedImageTools.crop(img, 0, 0, width, img.getHeight());
            } else if (img.getHeight() > height) {
                img = BufferedImageTools.crop(img, 0, 0, img.getWidth(), height);
            }
            if (height > img.getHeight() || width > img.getWidth()) {
                int padX = (int)Math.round((double)(width - img.getWidth()) * xProp);
                int padY = (int)Math.round((double)(height - img.getHeight()) * yProp);
                Padding padding = Padding.getPadding(padX, width - img.getWidth() - padX, padY, height - img.getHeight() - padY);
                img = TileExporter.pad(img, padding);
            }
        }
        return img;
    }

    private static BufferedImage readFixedSizeRegion(ImageServer<BufferedImage> server, RegionRequest request, int width, int height) throws IOException {
        BufferedImage img;
        double xProp = 0.0;
        double yProp = 0.0;
        if (request.getX() >= 0 && request.getY() >= 0 && request.getMaxX() <= server.getWidth() && request.getMaxY() <= server.getHeight()) {
            img = server.readRegion(request);
            if (img.getWidth() == width && img.getHeight() == height) {
                return img;
            }
            logger.warn("Requested {}x{}, got {}x{} for {}", new Object[]{width, height, img.getWidth(), img.getHeight(), request});
        } else {
            int x = GeneralTools.clipValue(request.getMinX(), 0, server.getWidth());
            int x2 = GeneralTools.clipValue(request.getMaxX(), 0, server.getWidth());
            int y = GeneralTools.clipValue(request.getMinY(), 0, server.getHeight());
            int y2 = GeneralTools.clipValue(request.getMaxY(), 0, server.getHeight());
            double downsample = request.getDownsample();
            RegionRequest request2 = RegionRequest.createInstance(server.getPath(), downsample, x, y, x2 - x, y2 - y, request.getImagePlane());
            img = server.readRegion(request2);
            if (height > img.getHeight() || width > img.getWidth()) {
                xProp = TileExporter.calculateFirstPadProportion(request.getMinX(), request.getMaxX(), 0.0, server.getWidth());
                yProp = TileExporter.calculateFirstPadProportion(request.getMinY(), request.getMaxY(), 0.0, server.getHeight());
            }
            img = TileExporter.cropOrPad(img, width, height, xProp, yProp);
        }
        boolean smoothInterpolate = true;
        if (img.getColorModel() instanceof IndexColorModel || server instanceof LabeledImageServer) {
            smoothInterpolate = false;
        }
        return BufferedImageTools.resize(img, width, height, smoothInterpolate);
    }

    private static BufferedImage pad(BufferedImage img, Padding padding) {
        if (padding.isEmpty()) {
            return img;
        }
        WritableRaster raster = img.getRaster();
        int width = img.getWidth() + padding.getXSum();
        int height = img.getHeight() + padding.getYSum();
        WritableRaster raster2 = raster.createCompatibleWritableRaster(width, height);
        raster2.setDataElements(padding.getX1(), padding.getY1(), raster);
        return new BufferedImage(img.getColorModel(), raster2, img.isAlphaPremultiplied(), null);
    }

    private static double calculateFirstPadProportion(double v1, double v2, double minVal, double maxVal) {
        if (v1 >= minVal) {
            return 0.0;
        }
        if (v2 <= maxVal) {
            return 1.0;
        }
        double d1 = minVal - v1;
        double d2 = v2 - maxVal;
        return d1 / (d1 + d2);
    }

    private static NumberFormat createDefaultNumberFormat(int maxFractionDigits) {
        NumberFormat formatter = NumberFormat.getNumberInstance(Locale.US);
        formatter.setMinimumFractionDigits(0);
        formatter.setMaximumFractionDigits(maxFractionDigits);
        return formatter;
    }

    static String getRegionString(RegionRequest request) {
        Object s = "";
        if (request.getDownsample() != 1.0) {
            s = "d=" + formatter.get().format(request.getDownsample()) + ",";
        }
        s = (String)s + "x=" + request.getX() + ",y=" + request.getY() + ",w=" + request.getWidth() + ",h=" + request.getHeight();
        if (request.getZ() != 0) {
            s = (String)s + ",z=" + request.getZ();
        }
        if (request.getT() != 0) {
            s = (String)s + ",t=" + request.getT();
        }
        return s;
    }

    Collection<RegionRequestWrapper> getTiledRegionRequests(double downsample) {
        ArrayList<RegionRequestWrapper> requests = new ArrayList<RegionRequestWrapper>();
        if (downsample == 0.0) {
            throw new IllegalArgumentException("No downsample was specified!");
        }
        ImageRegion regionLocal = this.region == null ? RegionRequest.createInstance(this.server, downsample) : this.region;
        int minZLocal = this.minZ < 0 ? 0 : this.minZ;
        int minTLocal = this.minT < 0 ? 0 : this.minT;
        int maxZLocal = this.maxZ > this.server.nZSlices() || this.maxZ == -1 ? this.server.nZSlices() : this.maxZ;
        int maxTLocal = this.maxT > this.server.nTimepoints() || this.maxT == -1 ? this.server.nTimepoints() : this.maxT;
        RegionRequest regionLocal2 = RegionRequest.createInstance(this.server.getPath(), downsample, regionLocal);
        for (int t = minTLocal; t < maxTLocal; ++t) {
            regionLocal2 = regionLocal2.updateT(t);
            for (int z = minZLocal; z < maxZLocal; ++z) {
                regionLocal2 = regionLocal2.updateZ(z);
                requests.addAll(TileExporter.splitRegionRequests(regionLocal2, this.tileWidth, this.tileHeight, this.overlapX, this.overlapY, this.includePartialTiles));
            }
        }
        return requests;
    }

    static Collection<RegionRequestWrapper> splitRegionRequests(RegionRequest request, int tileWidth, int tileHeight, int xOverlap, int yOverlap, boolean includePartialTiles) {
        if (tileWidth <= 0 || tileHeight <= 0) {
            throw new IllegalArgumentException(String.format("Unsupported tile size (%d x %d) - dimensions must be > 0", tileWidth, tileHeight));
        }
        if (xOverlap >= tileWidth || yOverlap >= tileHeight) {
            throw new IllegalArgumentException("Overlap must be less than the tile size!");
        }
        LinkedHashSet<RegionRequestWrapper> set = new LinkedHashSet<RegionRequestWrapper>();
        double downsample = request.getDownsample();
        String path = request.getPath();
        int minX = (int)Math.round((double)request.getMinX() / downsample);
        int minY = (int)Math.round((double)request.getMinY() / downsample);
        int maxX = (int)Math.round((double)request.getMaxX() / downsample);
        int maxY = (int)Math.round((double)request.getMaxY() / downsample);
        int z = request.getZ();
        int t = request.getT();
        for (int y = minY; y < maxY; y += tileHeight - yOverlap) {
            int th = tileHeight;
            if (y + th > maxY) {
                th = maxY - y;
            }
            boolean partialTile = false;
            int yi = (int)Math.round((double)y * downsample);
            int y2i = (int)Math.round((double)(y + tileHeight) * downsample);
            if (y2i > request.getMaxY()) {
                if (!includePartialTiles) continue;
                partialTile = true;
                y2i = request.getMaxY();
            } else if (y2i == yi) continue;
            for (int x = minX; x < maxX; x += tileWidth - xOverlap) {
                int tw = tileWidth;
                if (x + tw > maxX) {
                    tw = maxX - x;
                }
                int xi = (int)Math.round((double)x * downsample);
                int x2i = (int)Math.round((double)(x + tileWidth) * downsample);
                if (x2i > request.getMaxX()) {
                    if (!includePartialTiles) continue;
                    partialTile = true;
                    x2i = request.getMaxX();
                } else if (x2i == xi) continue;
                RegionRequest tile = RegionRequest.createInstance(path, downsample, xi, yi, x2i - xi, y2i - yi, z, t);
                set.add(new RegionRequestWrapper(tile, partialTile));
            }
        }
        return set;
    }

    static class RegionRequestWrapper {
        final RegionRequest request;
        final boolean partialTile;

        RegionRequestWrapper(RegionRequest request, boolean partialTile) {
            this.request = request;
            this.partialTile = partialTile;
        }
    }

    static class ExportTask
    implements Runnable {
        private ImageServer<BufferedImage> server;
        private RegionRequest request;
        private String path;
        private int tileWidth;
        private int tileHeight;
        private boolean ensureSize;

        private ExportTask(ImageServer<BufferedImage> server, RegionRequest request, String path, int tileWidth, int tileHeight, boolean ensureSize) {
            this.server = server;
            this.request = request;
            this.path = path;
            this.tileWidth = tileWidth;
            this.tileHeight = tileHeight;
            this.ensureSize = ensureSize;
        }

        @Override
        public void run() {
            try {
                if (Thread.currentThread().isInterrupted()) {
                    logger.debug("Interrupted! Will not write image to {}", (Object)this.path);
                    return;
                }
                if (this.ensureSize) {
                    BufferedImage img = TileExporter.readFixedSizeRegion(this.server, this.request, this.tileWidth, this.tileHeight);
                    ImageWriterTools.writeImage(img, this.path);
                } else {
                    ImageWriterTools.writeImageRegion(this.server, this.request, this.path);
                }
            }
            catch (Exception e) {
                logger.error("Error writing tile: " + e.getLocalizedMessage(), (Throwable)e);
            }
        }
    }

    private static class TileExportEntry {
        private RegionRequest region;
        private String image;
        private String labels;

        TileExportEntry(RegionRequest region, String image, String labels) {
            this.region = region;
            this.image = image;
            this.labels = labels;
        }
    }

    private static class TileExportData {
        private String qupath_version = GeneralTools.getVersion();
        private String base_directory;
        private List<TileExportLabel> labels;
        private List<TileExportEntry> tiles;

        TileExportData(String path, List<TileExportEntry> images) {
            this.base_directory = path;
            this.tiles = images;
        }
    }

    private static class TileExportLabel {
        private String classification;
        private Integer label;
        private Integer boundaryLabel;

        public TileExportLabel(String classification, Integer label, Integer boundaryLabel) {
            this.classification = classification;
            this.label = label;
            this.boundaryLabel = boundaryLabel;
        }
    }
}

