/*
 * Decompiled with CFR 0.152.
 */
package qupath.lib.objects;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import org.locationtech.jts.algorithm.Centroid;
import org.locationtech.jts.geom.Coordinate;
import org.locationtech.jts.geom.Envelope;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.GeometryCollection;
import org.locationtech.jts.geom.Point;
import org.locationtech.jts.geom.Polygon;
import org.locationtech.jts.geom.util.AffineTransformation;
import org.locationtech.jts.geom.util.GeometryFixer;
import org.locationtech.jts.index.strtree.STRtree;
import org.locationtech.jts.simplify.VWSimplifier;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import qupath.lib.analysis.DelaunayTools;
import qupath.lib.objects.PathCellObject;
import qupath.lib.objects.PathObject;
import qupath.lib.objects.PathObjectTools;
import qupath.lib.objects.PathObjects;
import qupath.lib.roi.GeometryTools;
import qupath.lib.roi.interfaces.ROI;

public class CellTools {
    private static final Logger logger = LoggerFactory.getLogger(CellTools.class);

    public static PathCellObject constrainCellByScaledNucleus(PathCellObject cell, double nucleusScaleFactor, boolean keepMeasurements) {
        ROI roi = cell.getROI();
        ROI roiNucleus = cell.getNucleusROI();
        if (roi == null || roiNucleus == null) {
            return cell;
        }
        Geometry geom = roi.getGeometry();
        Geometry geomNucleus = roiNucleus.getGeometry();
        Point centroid = geomNucleus.getCentroid();
        AffineTransformation transform = AffineTransformation.scaleInstance((double)nucleusScaleFactor, (double)nucleusScaleFactor, (double)centroid.getX(), (double)centroid.getY());
        Geometry geomNucleusExpanded = transform.transform(geomNucleus);
        if (geomNucleusExpanded.covers(geom)) {
            return cell;
        }
        geom = geom.intersection(geomNucleusExpanded);
        geom = GeometryTools.ensurePolygonal(geom);
        roi = GeometryTools.geometryToROI(geom, roi.getImagePlane());
        return (PathCellObject)PathObjects.createCellObject(roi, roiNucleus, cell.getPathClass(), keepMeasurements ? cell.getMeasurementList() : null);
    }

    public static List<PathObject> constrainCellOverlaps(Collection<? extends PathObject> cells) {
        LinkedHashMap<PathObject, Geometry> map = new LinkedHashMap<PathObject, Geometry>();
        for (PathObject pathObject : cells) {
            if (!pathObject.isCell()) {
                logger.warn("{} is not a cell - will be skipped!", (Object)pathObject);
                continue;
            }
            map.put(pathObject, pathObject.getROI().getGeometry());
        }
        return CellTools.detectionsToCells(map);
    }

    public static List<PathObject> detectionsToCells(Collection<? extends PathObject> detections, double distance, double nucleusScale) {
        AffineTransformation transform = new AffineTransformation();
        LinkedHashMap<PathObject, Geometry> map = new LinkedHashMap<PathObject, Geometry>();
        for (PathObject pathObject : detections) {
            ROI roiNucleus = PathObjectTools.getROI(pathObject, true);
            Geometry geomNucleus = roiNucleus.getGeometry();
            Geometry geomCell = CellTools.estimateCellBoundary(geomNucleus, distance, nucleusScale, transform);
            map.put(pathObject, geomCell);
        }
        return CellTools.detectionsToCells(map);
    }

    public static Geometry estimateCellBoundary(Geometry geomNucleus, double distance, double nucleusScale) {
        return CellTools.estimateCellBoundary(geomNucleus, distance, nucleusScale, new AffineTransformation());
    }

    private static Geometry estimateCellBoundary(Geometry geomNucleus, double distance, double nucleusScale, AffineTransformation transform) {
        Geometry geomCell = geomNucleus.buffer(distance);
        if (nucleusScale > 1.0) {
            Geometry geomNucleusHull = geomNucleus.convexHull();
            Coordinate centroid = new Centroid(geomNucleusHull).getCentroid();
            double x = centroid.getX();
            double y = centroid.getY();
            transform.setToTranslation(-x, -y);
            transform.scale(nucleusScale, nucleusScale);
            transform.translate(x, y);
            Geometry geomNucleusExpanded = transform.transform(geomNucleusHull);
            if (!geomNucleusExpanded.covers(geomCell)) {
                geomCell = GeometryTools.attemptOperation(geomCell, g -> g.intersection(geomNucleusExpanded));
            }
            if (geomNucleusHull.getNumGeometries() == 1 && geomCell.getNumGeometries() == 1 && !geomCell.covers(geomNucleusHull)) {
                geomCell = GeometryTools.attemptOperation(geomCell, g -> g.union(geomNucleusHull));
            }
        }
        if (geomCell instanceof GeometryCollection && geomNucleus instanceof Polygon && !geomCell.isValid()) {
            geomCell = GeometryFixer.fix((Geometry)geomCell);
            logger.debug("Used GeometryFixer to fix an invalid cell boundary geometry");
        }
        return geomCell;
    }

    private static List<PathObject> detectionsToCells(Map<PathObject, Geometry> cellBoundaryMap) {
        int max = 500;
        if (cellBoundaryMap.size() > max) {
            STRtree cache = new STRtree(max);
            HashMap<PathObject, Envelope> envelopes = new HashMap<PathObject, Envelope>();
            HashMap<PathObject, Geometry> geometries = new HashMap<PathObject, Geometry>();
            HashMap<PathObject, Geometry> nucleusGeometries = new HashMap<PathObject, Geometry>();
            for (Map.Entry<PathObject, Geometry> entry : cellBoundaryMap.entrySet()) {
                PathObject detection = entry.getKey();
                ROI roi = PathObjectTools.getROI(detection, true);
                Geometry geomNucleus = roi.getGeometry();
                Geometry geomCell = entry.getValue();
                Envelope env = geomCell.getEnvelopeInternal();
                envelopes.put(detection, env);
                geometries.put(detection, geomCell);
                nucleusGeometries.put(detection, geomNucleus);
                cache.insert(env, (Object)detection);
            }
            cache.build();
            List items = cache.itemsTree();
            return CellTools.detectionsToCellsSubtree(cache, items, cellBoundaryMap, envelopes);
        }
        return CellTools.detectionsToCells(cellBoundaryMap.keySet(), cellBoundaryMap.keySet(), cellBoundaryMap);
    }

    private static List<PathObject> detectionsToCellsSubtree(STRtree tree, List<?> list, Map<PathObject, Geometry> cellBoundaryMap, Map<PathObject, Envelope> envelopes) {
        if (list.isEmpty()) {
            return Collections.emptyList();
        }
        PathObject first = list.get(0);
        boolean doParallel = true;
        if (first instanceof PathObject) {
            Envelope env = new Envelope();
            List<PathObject> pathObjects = list;
            for (PathObject pathObject : pathObjects) {
                env.expandToInclude(envelopes.get(pathObject));
            }
            List list2 = tree.query(env);
            try {
                return CellTools.detectionsToCells(pathObjects, list2, cellBoundaryMap);
            }
            catch (Exception e) {
                logger.error("Error converting detections to cells: " + e.getLocalizedMessage(), (Throwable)e);
                return Collections.emptyList();
            }
        }
        if (first instanceof List) {
            if (doParallel) {
                return list.parallelStream().map(sublist -> CellTools.detectionsToCellsSubtree(tree, (List)sublist, cellBoundaryMap, envelopes)).flatMap(Collection::stream).toList();
            }
            ArrayList<PathObject> cells = new ArrayList<PathObject>();
            for (PathObject pathObject : list) {
                cells.addAll(CellTools.detectionsToCellsSubtree(tree, (List)((Object)pathObject), cellBoundaryMap, envelopes));
            }
            return cells;
        }
        throw new IllegalArgumentException("Expected a list of PathObjects or a list of lists, but got a list of " + String.valueOf(first.getClass()));
    }

    private static List<PathObject> detectionsToCells(Collection<PathObject> detections, Collection<PathObject> allDetections, Map<PathObject, Geometry> cellBoundaryMap) {
        DelaunayTools.Subdivision subdivision = DelaunayTools.newBuilder(allDetections).preferNucleus(true).roiBounds(2.0, 1.0).build();
        ArrayList<PathObject> cells = new ArrayList<PathObject>();
        Map<PathObject, Geometry> faces = subdivision.getVoronoiFaces();
        for (PathObject detection : detections) {
            Geometry face = faces.get(detection);
            Geometry bounds = cellBoundaryMap.get(detection);
            if (face == null || bounds == null) {
                logger.warn("Missing boundary information for {} - will skip", (Object)detection);
                continue;
            }
            try {
                ROI roiNucleus;
                Geometry geomCell = face;
                try {
                    geomCell = face.intersection(bounds);
                    geomCell = GeometryTools.ensurePolygonal(geomCell);
                    geomCell = VWSimplifier.simplify((Geometry)geomCell, (double)1.0);
                }
                catch (Exception e) {
                    if (face.getArea() > bounds.getArea()) {
                        geomCell = bounds;
                        logger.warn("Error computing intersection between cell boundary and Voronoi face - will use bounds result: " + e.getLocalizedMessage(), (Throwable)e);
                    }
                    logger.warn("Error computing intersection between cell boundary and Voronoi face - will use Voronoi result: " + e.getLocalizedMessage(), (Throwable)e);
                }
                ROI roiCell = roiNucleus = PathObjectTools.getROI(detection, true);
                if (geomCell.isEmpty()) {
                    logger.warn("Unable to create cell ROI for {} - I'll use the nucleus ROI instead", (Object)detection);
                } else {
                    Geometry geomNucleus = roiNucleus.getGeometry();
                    if (!geomCell.covers(geomNucleus)) {
                        try {
                            geomNucleus = geomCell.intersection(geomNucleus);
                            roiNucleus = GeometryTools.geometryToROI(geomNucleus, roiNucleus.getImagePlane());
                        }
                        catch (Exception e) {
                            logger.debug("Error constraining nucleus to cell: {}", (Object)e.getLocalizedMessage());
                        }
                    }
                    roiCell = GeometryTools.geometryToROI(geomCell, roiNucleus.getImagePlane());
                }
                cells.add(PathObjects.createCellObject(roiCell, roiNucleus, detection.getPathClass(), detection.getMeasurementList()));
            }
            catch (Exception ex) {
                logger.warn("Exception creating cell for {} - will skip", (Object)PathObjectTools.getROI(detection, true));
            }
        }
        return cells;
    }

    public static PathCellObject constrainCellByNucleusDistance(PathCellObject cell, double distance, boolean keepMeasurements) {
        ROI roi = cell.getROI();
        ROI roiNucleus = cell.getNucleusROI();
        if (roi == null || roiNucleus == null || distance <= 0.0) {
            return cell;
        }
        Geometry geom = roi.getGeometry();
        Geometry geomNucleus = roiNucleus.getGeometry().convexHull();
        Geometry geomNucleusExpanded = geomNucleus.buffer(distance);
        if (geomNucleusExpanded.covers(geom)) {
            return cell;
        }
        geom = geom.intersection(geomNucleusExpanded);
        geom = GeometryTools.ensurePolygonal(geom);
        roi = GeometryTools.geometryToROI(geom, roi.getImagePlane());
        return (PathCellObject)PathObjects.createCellObject(roi, roiNucleus, cell.getPathClass(), keepMeasurements ? cell.getMeasurementList() : null);
    }
}

