/*
 * Decompiled with CFR 0.152.
 */
package qupath.lib.objects;

import java.awt.geom.AffineTransform;
import java.awt.geom.Line2D;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.UUID;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.prep.PreparedGeometry;
import org.locationtech.jts.geom.prep.PreparedGeometryFactory;
import org.locationtech.jts.index.strtree.STRtree;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import qupath.lib.geom.Point2;
import qupath.lib.images.ImageData;
import qupath.lib.images.servers.ImageServer;
import qupath.lib.measurements.MeasurementList;
import qupath.lib.objects.PathAnnotationObject;
import qupath.lib.objects.PathCellObject;
import qupath.lib.objects.PathDetectionObject;
import qupath.lib.objects.PathObject;
import qupath.lib.objects.PathObjects;
import qupath.lib.objects.PathROIObject;
import qupath.lib.objects.PathRootObject;
import qupath.lib.objects.PathTileObject;
import qupath.lib.objects.TMACoreObject;
import qupath.lib.objects.classes.PathClass;
import qupath.lib.objects.classes.PathClassTools;
import qupath.lib.objects.hierarchy.DefaultTMAGrid;
import qupath.lib.objects.hierarchy.PathObjectHierarchy;
import qupath.lib.objects.hierarchy.TMAGrid;
import qupath.lib.regions.ImagePlane;
import qupath.lib.regions.ImageRegion;
import qupath.lib.regions.RegionRequest;
import qupath.lib.roi.GeometryTools;
import qupath.lib.roi.LineROI;
import qupath.lib.roi.PointsROI;
import qupath.lib.roi.ROIs;
import qupath.lib.roi.RoiTools;
import qupath.lib.roi.interfaces.ROI;

public class PathObjectTools {
    private static final Logger logger = LoggerFactory.getLogger(PathObjectTools.class);

    private static void removePoints(Collection<PathObject> pathObjects) {
        logger.trace("Remove points");
        pathObjects.removeIf(PathObjectTools::hasPointROI);
    }

    public static boolean hasPointROI(PathObject pathObject) {
        return pathObject.hasROI() && pathObject.getROI().isPoint();
    }

    public static int countObjectsWithClass(Collection<? extends PathObject> pathObjects, PathClass pathClass, boolean useBaseClass) {
        int count = 0;
        for (PathObject pathObject : pathObjects) {
            PathClass currentClass = pathObject.getPathClass();
            if (useBaseClass) {
                PathClass pathClass2 = currentClass = currentClass == null ? null : currentClass.getBaseClass();
            }
            if (!Objects.equals(pathClass, currentClass)) continue;
            ++count;
        }
        return count;
    }

    public static List<PathObject> getObjectsOfClass(Collection<PathObject> pathObjects, Class<? extends PathObject> cls) {
        logger.trace("Get objects of class {}", cls);
        ArrayList<PathObject> pathObjectsFiltered = new ArrayList<PathObject>(pathObjects.size());
        for (PathObject temp : pathObjects) {
            if (cls != null && !cls.isInstance(temp)) continue;
            pathObjectsFiltered.add(temp);
        }
        return pathObjectsFiltered;
    }

    public static Predicate<PathObject> createImageRegionPredicate(ImageRegion region) {
        return new ImageRegionPredicate(region);
    }

    public static <T extends PathObject> List<T> getLineObjects(Collection<T> pathObjects) {
        return pathObjects.stream().filter(p -> p.getROI() != null && p.getROI().isLine()).toList();
    }

    public static <T extends PathObject> List<T> getAreaObjects(Collection<T> pathObjects) {
        return pathObjects.stream().filter(p -> p.getROI() != null && p.getROI().isArea()).toList();
    }

    public static <T extends PathObject> List<T> getPointObjects(Collection<T> pathObjects) {
        return pathObjects.stream().filter(p -> p.getROI() != null && p.getROI().isPoint()).toList();
    }

    public static List<PathObject> getFlattenedObjectList(PathObject parentObject, List<PathObject> list, boolean includeParent) {
        if (list == null) {
            list = new ArrayList<PathObject>();
        }
        if (includeParent) {
            list.add(parentObject);
        }
        for (PathObject child : parentObject.getChildObjectsAsArray()) {
            PathObjectTools.getFlattenedObjectList(child, list, true);
        }
        return list;
    }

    public static int countDescendants(PathObject pathObject) {
        return pathObject.nDescendants();
    }

    public static List<PathObject> findObjectsOutsideImage(Collection<? extends PathObject> pathObjects, ImageServer<?> server, boolean ignoreIntersecting) {
        return PathObjectTools.findObjectsOutsideRegion(pathObjects, RegionRequest.createInstance(server), 0, server.nZSlices(), 0, server.nTimepoints(), ignoreIntersecting);
    }

    public static List<PathObject> findObjectsOutsideRegion(Collection<? extends PathObject> pathObjects, ImageRegion region, boolean ignoreIntersecting) {
        return PathObjectTools.findObjectsOutsideRegion(pathObjects, region, region.getZ(), region.getZ() + 1, region.getT(), region.getT() + 1, ignoreIntersecting);
    }

    public static List<PathObject> findObjectsOutsideRegion(Collection<? extends PathObject> pathObjects, ImageRegion region, int minZ, int maxZ, int minT, int maxT, boolean ignoreIntersecting) {
        return pathObjects.stream().filter(p -> !PathObjectTools.checkRegionContainsROI(p.getROI(), region, minZ, maxZ, minT, maxT, ignoreIntersecting)).collect(Collectors.toList());
    }

    public static <T extends PathObject> Collection<T> filterByROICovers(ROI roi, Collection<T> pathObjects) {
        return PathObjectTools.filterByROICovers(roi, pathObjects, PathObject::getROI);
    }

    public static <T extends PathObject> Collection<T> filterByROICoversNucleus(ROI roi, Collection<T> pathObjects) {
        return PathObjectTools.filterByROICovers(roi, pathObjects, PathObjectTools::getNucleusOrMainROI);
    }

    private static <T extends PathObject> Collection<T> filterByROICovers(ROI roi, Collection<T> pathObjects, Function<PathObject, ROI> roiExtractor) {
        Predicate<PathObject> predicate;
        Geometry geom = roi.getGeometry();
        if (pathObjects.size() > 1) {
            PreparedGeometry prepared = PreparedGeometryFactory.prepare((Geometry)geom);
            predicate = PathObjectTools.createGeometryPredicate(arg_0 -> ((PreparedGeometry)prepared).covers(arg_0), roiExtractor);
        } else {
            predicate = PathObjectTools.createGeometryPredicate(arg_0 -> ((Geometry)geom).covers(arg_0), roiExtractor);
        }
        return PathObjectTools.filterByROIPredicate(roi, predicate, pathObjects);
    }

    public static <T extends PathObject> Collection<T> filterByROIIntersects(ROI roi, Collection<T> pathObjects) {
        return PathObjectTools.filterByROIIntersects(roi, pathObjects, PathObject::getROI);
    }

    public static <T extends PathObject> Collection<T> filterByROIIntersectsNucleus(ROI roi, Collection<T> pathObjects) {
        return PathObjectTools.filterByROIIntersects(roi, pathObjects, PathObjectTools::getNucleusOrMainROI);
    }

    private static <T extends PathObject> Collection<T> filterByROIIntersects(ROI roi, Collection<T> pathObjects, Function<PathObject, ROI> roiExtractor) {
        Predicate<PathObject> predicate;
        Geometry geom = roi.getGeometry();
        if (pathObjects.size() > 1) {
            PreparedGeometry prepared = PreparedGeometryFactory.prepare((Geometry)geom);
            predicate = PathObjectTools.createGeometryPredicate(arg_0 -> ((PreparedGeometry)prepared).intersects(arg_0), roiExtractor);
        } else {
            predicate = PathObjectTools.createGeometryPredicate(arg_0 -> ((Geometry)geom).intersects(arg_0), roiExtractor);
        }
        return PathObjectTools.filterByROIPredicate(roi, predicate, pathObjects);
    }

    private static <T extends PathObject> Collection<T> filterByROIPredicate(ROI roi, Predicate<PathObject> predicate, Collection<T> pathObjects) {
        Predicate<PathObject> planePredicate = PathObjectTools.createPlanePredicate(roi.getImagePlane());
        return pathObjects.parallelStream().filter(PathObject::hasROI).filter(planePredicate).filter(predicate).collect(Collectors.toList());
    }

    private static Predicate<PathObject> createPlanePredicate(ImagePlane plane) {
        return p -> p.getROI().getZ() == plane.getZ() && p.getROI().getT() == plane.getT();
    }

    private static Predicate<PathObject> createGeometryPredicate(Predicate<Geometry> predicate, Function<PathObject, ROI> roiFunction) {
        return p -> predicate.test(((ROI)roiFunction.apply((PathObject)p)).getGeometry());
    }

    public static <T extends PathObject> Collection<T> filterByROIContainsCentroid(ROI roi, Collection<T> pathObjects) {
        return PathObjectTools.filterByROIPredicate(roi, p -> roi.contains(p.getROI().getCentroidX(), p.getROI().getCentroidY()), pathObjects);
    }

    public static <T extends PathObject> Collection<T> filterByROIContainsNucleusCentroid(ROI roi, Collection<T> pathObjects) {
        return PathObjectTools.filterByROIPredicate(roi, p -> roi.contains(PathObjectTools.getNucleusOrMainROI(p).getCentroidX(), PathObjectTools.getNucleusOrMainROI(p).getCentroidY()), pathObjects);
    }

    private static boolean checkRegionContainsROI(ROI roi, ImageRegion region, int minZ, int maxZ, int minT, int maxT, boolean ignoreIntersecting) {
        boolean regionContainsBounds;
        if (roi == null) {
            return false;
        }
        if (roi.getZ() < minZ || roi.getZ() >= maxZ || roi.getT() < minT || roi.getT() >= maxT) {
            return false;
        }
        boolean bl = regionContainsBounds = (double)region.getX() <= roi.getBoundsX() && (double)region.getY() <= roi.getBoundsY() && (double)region.getMaxX() >= roi.getBoundsX() + roi.getBoundsWidth() && (double)region.getMaxY() >= roi.getBoundsY() + roi.getBoundsHeight();
        if (regionContainsBounds) {
            return true;
        }
        if (ignoreIntersecting) {
            return RoiTools.intersectsRegion(roi.updatePlane(region.getImagePlane()), region);
        }
        return false;
    }

    public static PathObject updatePlaneRecursive(PathObject pathObject, ImagePlane plane, boolean copyMeasurements, boolean createNewIDs) {
        PathObject newObj = PathObjectTools.transformObjectImpl(pathObject, r -> r.updatePlane(plane), copyMeasurements, createNewIDs);
        if (pathObject.hasChildObjects()) {
            List<PathObject> newChildObjects = pathObject.getChildObjects().parallelStream().map(p -> PathObjectTools.updatePlaneRecursive(p, plane, copyMeasurements, createNewIDs)).toList();
            newObj.addChildObjects(newChildObjects);
        }
        return newObj;
    }

    public static PathObject updatePlaneRecursive(PathObject pathObject, ImagePlane plane) {
        return PathObjectTools.updatePlaneRecursive(pathObject, plane, false, true);
    }

    public static PathObject updatePlane(PathObject pathObject, ImagePlane plane, boolean copyMeasurements, boolean createNewIDs) {
        return PathObjectTools.transformObjectImpl(pathObject, r -> r.updatePlane(plane), copyMeasurements, createNewIDs);
    }

    public static String getSuitableName(Class<? extends PathObject> cls, boolean makePlural) {
        if (makePlural) {
            if (cls.equals(PathRootObject.class)) {
                return "Root objects";
            }
            if (cls.equals(PathAnnotationObject.class)) {
                return "Annotations";
            }
            if (cls.equals(TMACoreObject.class)) {
                return "TMA cores";
            }
            if (cls.equals(PathDetectionObject.class)) {
                return "Detections";
            }
            if (cls.equals(PathCellObject.class)) {
                return "Cells";
            }
            if (cls.equals(PathTileObject.class)) {
                return "Tiles";
            }
            return cls.getSimpleName() + " objects";
        }
        if (cls.equals(PathRootObject.class)) {
            return "Root object";
        }
        if (cls.equals(PathAnnotationObject.class)) {
            return "Annotation";
        }
        if (cls.equals(TMACoreObject.class)) {
            return "TMA core";
        }
        if (cls.equals(PathDetectionObject.class)) {
            return "Detection";
        }
        if (cls.equals(PathCellObject.class)) {
            return "Cell";
        }
        if (cls.equals(PathTileObject.class)) {
            return "Tile";
        }
        return cls.getSimpleName();
    }

    public static boolean isAncestor(PathObject pathObject, PathObject possibleAncestor) {
        for (PathObject parent = pathObject.getParent(); parent != null; parent = parent.getParent()) {
            if (!parent.equals(possibleAncestor)) continue;
            return true;
        }
        return false;
    }

    public static List<TMACoreObject> getTMACoreObjects(PathObjectHierarchy hierarchy, boolean includeMissingCores) {
        TMAGrid tmaGrid = hierarchy.getTMAGrid();
        if (tmaGrid == null || tmaGrid.nCores() == 0) {
            return Collections.emptyList();
        }
        if (includeMissingCores) {
            return tmaGrid.getTMACoreList();
        }
        ArrayList<TMACoreObject> cores = new ArrayList<TMACoreObject>();
        for (TMACoreObject core : tmaGrid.getTMACoreList()) {
            if (core.isMissing()) continue;
            cores.add(core);
        }
        return cores;
    }

    public static TMACoreObject getAncestorTMACore(PathObject pathObject) {
        PathObject parent;
        for (parent = pathObject; parent != null && !(parent instanceof TMACoreObject); parent = parent.getParent()) {
        }
        return (TMACoreObject)parent;
    }

    public static TMACoreObject getTMACoreForPixel(TMAGrid tmaGrid, double x, double y) {
        return PathObjectTools.getPathObjectContainingPixel(tmaGrid.getTMACoreList(), x, y);
    }

    private static <T extends PathObject> T getPathObjectContainingPixel(Collection<T> pathObjects, double x, double y) {
        for (PathObject pathObject : pathObjects) {
            if (!RoiTools.areaContains(pathObject.getROI(), x, y)) continue;
            return (T)pathObject;
        }
        return null;
    }

    public static void addTMAGrid(ImageData<?> imageData, String hLabels, String vLabels, boolean rowFirst, double diameterCalibrated) {
        double diameterPixels = diameterCalibrated / imageData.getServer().getPixelCalibration().getAveragedPixelSize().doubleValue();
        PathObjectHierarchy hierarchy = imageData.getHierarchy();
        PathObject selected = hierarchy.getSelectionModel().getSelectedObject();
        ROI roi = selected == null ? null : selected.getROI();
        ImageRegion region = roi == null ? ImageRegion.createInstance(0, 0, imageData.getServer().getWidth(), imageData.getServer().getHeight(), 0, 0) : ImageRegion.createInstance(roi);
        TMAGrid tmaGrid = PathObjectTools.createTMAGrid(hLabels, vLabels, rowFirst, diameterPixels, region);
        hierarchy.setTMAGrid(tmaGrid);
    }

    public static TMAGrid createTMAGrid(String hLabels, String vLabels, boolean rowFirst, double diameterPixels, ImageRegion region) {
        String[] hLabelsSplit = PathObjectTools.parseTMALabelString(hLabels);
        String[] vLabelsSplit = PathObjectTools.parseTMALabelString(vLabels);
        int numHorizontal = hLabelsSplit.length;
        int numVertical = vLabelsSplit.length;
        ArrayList<TMACoreObject> cores = new ArrayList<TMACoreObject>();
        double xSpacing = ((double)region.getWidth() - diameterPixels) / (double)Math.max(1, numHorizontal - 1);
        double ySpacing = ((double)region.getHeight() - diameterPixels) / (double)Math.max(1, numVertical - 1);
        for (int i = 0; i < numVertical; ++i) {
            for (int j = 0; j < numHorizontal; ++j) {
                double x = numHorizontal <= 1 ? (double)region.getMinX() + (double)region.getWidth() / 2.0 : (double)region.getMinX() + diameterPixels / 2.0 + xSpacing * (double)j;
                double y = numVertical <= 1 ? (double)region.getMinY() + (double)region.getHeight() / 2.0 : (double)region.getMinY() + diameterPixels / 2.0 + ySpacing * (double)i;
                cores.add(PathObjects.createTMACoreObject(x, y, diameterPixels, false, region.getImagePlane()));
            }
        }
        TMAGrid grid = DefaultTMAGrid.create(cores, numHorizontal);
        PathObjectTools.relabelTMAGrid(grid, hLabels, vLabels, rowFirst);
        return grid;
    }

    public static boolean relabelTMAGrid(TMAGrid grid, String labelsHorizontal, String labelsVertical, boolean rowFirst) {
        String[] columnLabels = PathObjectTools.parseTMALabelString(labelsHorizontal);
        String[] rowLabels = PathObjectTools.parseTMALabelString(labelsVertical);
        if (columnLabels.length < grid.getGridWidth()) {
            logger.error("Cannot relabel full TMA grid - not enough column labels specified!");
            return false;
        }
        if (rowLabels.length < grid.getGridHeight()) {
            logger.error("Cannot relabel full TMA grid - not enough row labels specified!");
            return false;
        }
        for (int r = 0; r < grid.getGridHeight(); ++r) {
            for (int c = 0; c < grid.getGridWidth(); ++c) {
                String name = rowFirst ? rowLabels[r] + "-" + columnLabels[c] : columnLabels[c] + "-" + rowLabels[r];
                grid.getTMACore(r, c).setName(name);
            }
        }
        return true;
    }

    public static void convertToPoints(PathObjectHierarchy hierarchy, Collection<PathObject> pathObjects, boolean preferNucleus, boolean deleteObjects) {
        Collection<PathObject> points = PathObjectTools.convertToPoints(pathObjects, preferNucleus);
        if (deleteObjects) {
            hierarchy.removeObjects(pathObjects, true);
        }
        hierarchy.addObjects(points);
    }

    public static Collection<PathObject> convertToPoints(Collection<PathObject> pathObjects, boolean preferNucleus) {
        HashMap<PathClass, Map> pointsMap = new HashMap<PathClass, Map>();
        for (PathObject pathObject : pathObjects) {
            ROI roi = PathObjectTools.getROI(pathObject, preferNucleus);
            if (roi == null) continue;
            ImagePlane plane = roi.getImagePlane();
            PathClass pathClass = pathObject.getPathClass();
            Map pointsMapByClass = pointsMap.computeIfAbsent(pathClass, p -> new HashMap());
            List points = pointsMapByClass.computeIfAbsent(plane, p -> new ArrayList());
            points.add(new Point2(roi.getCentroidX(), roi.getCentroidY()));
        }
        ArrayList<PathObject> annotations = new ArrayList<PathObject>();
        for (Map.Entry entry : pointsMap.entrySet()) {
            PathClass pathClass = (PathClass)entry.getKey();
            for (Map.Entry entry2 : ((Map)entry.getValue()).entrySet()) {
                ImagePlane plane = (ImagePlane)entry2.getKey();
                List points = (List)entry2.getValue();
                PathObject pointObject = PathObjects.createAnnotationObject(ROIs.createPointsROI(points, plane), pathClass);
                annotations.add(pointObject);
            }
        }
        return annotations;
    }

    public static boolean hierarchyContainsObject(PathObjectHierarchy hierarchy, PathObject pathObject) {
        PathObject testObject;
        if (pathObject == null) {
            return false;
        }
        for (testObject = pathObject; testObject != null && !(testObject instanceof PathRootObject); testObject = testObject.getParent()) {
        }
        return testObject == hierarchy.getRootObject();
    }

    public static Collection<PathObject> getObjectsForLocation(PathObjectHierarchy hierarchy, double x, double y, int zPos, int tPos, double vertexDistance) {
        if (hierarchy == null) {
            return Collections.emptyList();
        }
        HashSet<PathObject> pathObjects = new HashSet<PathObject>(8);
        int searchWidth = (int)Math.ceil(Math.max(vertexDistance * 2.0, 2.0));
        hierarchy.getAllObjectsForRegion(ImageRegion.createInstance((int)(x - (double)(searchWidth / 2)), (int)(y - (double)(searchWidth / 2)), searchWidth, searchWidth, zPos, tPos), pathObjects);
        if (vertexDistance < 0.0) {
            PathObjectTools.removePoints(pathObjects);
        }
        Iterator iter = pathObjects.iterator();
        double distSq = vertexDistance * vertexDistance;
        while (iter.hasNext()) {
            PathObject temp = (PathObject)iter.next();
            ROI roi = temp.getROI();
            if (RoiTools.areaContains(temp.getROI(), x, y)) continue;
            if (!roi.isArea() && vertexDistance >= 0.0) {
                boolean isClose = false;
                if (roi instanceof LineROI) {
                    LineROI line = (LineROI)roi;
                    if (Line2D.ptSegDistSq(line.getX1(), line.getY1(), line.getX2(), line.getY2(), x, y) <= distSq) {
                        isClose = true;
                    }
                } else if (roi.isLine()) {
                    Point2 lastPoint = null;
                    for (Point2 p : temp.getROI().getAllPoints()) {
                        if (p.distanceSq(x, y) <= distSq || lastPoint != null && Line2D.ptSegDistSq(p.getX(), p.getY(), lastPoint.getX(), lastPoint.getY(), x, y) <= distSq) {
                            isClose = true;
                            break;
                        }
                        lastPoint = p;
                    }
                } else {
                    for (Point2 p : temp.getROI().getAllPoints()) {
                        if (!(p.distanceSq(x, y) <= distSq)) continue;
                        isClose = true;
                        break;
                    }
                }
                if (isClose) continue;
            }
            iter.remove();
        }
        if (pathObjects.isEmpty()) {
            return Collections.emptyList();
        }
        return pathObjects;
    }

    public static List<PathObject> getAncestorList(PathObject pathObject) {
        ArrayList<PathObject> ancestors = new ArrayList<PathObject>();
        for (PathObject parent = pathObject; parent != null; parent = parent.getParent()) {
            ancestors.add(0, parent);
        }
        return ancestors;
    }

    public static void swapNameAndClass(PathObject pathObject, boolean includeColor) {
        PathClass pathClass = pathObject.getPathClass();
        String name = pathObject.getName();
        Integer color = pathObject.getColor();
        if (name == null) {
            pathObject.resetPathClass();
        } else {
            pathObject.setPathClass(PathClass.fromString(name, color));
        }
        if (pathClass == null) {
            pathObject.setName(null);
            if (includeColor) {
                pathObject.setColor(null);
            }
        } else {
            pathObject.setName(pathClass.toString());
            if (includeColor) {
                pathObject.setColor(pathClass.getColor());
            }
        }
    }

    public static String[] parseTMALabelString(String labelString) {
        if (labelString == null || labelString.length() == 0) {
            return new String[0];
        }
        String[] labels = (labelString = labelString.trim()).split(" ");
        if (labels.length == 1 && labels[0].contains("-")) {
            String[] labelsSplit = labels[0].split("-");
            try {
                int i1 = Integer.parseInt(labelsSplit[0]);
                int i2 = Integer.parseInt(labelsSplit[1]);
                int inc = 1;
                int n = i2 - i1 + 1;
                if (i1 > i2) {
                    inc = -1;
                    n = i1 - i2 + 1;
                }
                Object format = "%d";
                if (labelsSplit[0].startsWith("0")) {
                    format = "%0" + labelsSplit[0].length() + "d";
                }
                labels = new String[n];
                for (int i = 0; i < n; ++i) {
                    labels[i] = String.format((String)format, i1 + i * inc);
                }
                return labels;
            }
            catch (Exception i1) {
                try {
                    char c1 = labelsSplit[0].charAt(0);
                    char c2 = labelsSplit[1].charAt(0);
                    int inc = 1;
                    int n = c2 - c1 + 1;
                    if (c1 > c2) {
                        inc = -1;
                        n = c1 - c2 + 1;
                    }
                    labels = new String[n];
                    int counter = 0;
                    for (int i = 0; i < n; i = (int)((char)(i + 1))) {
                        labels[counter] = "" + (char)(c1 + i * inc);
                        ++counter;
                    }
                    return labels;
                }
                catch (Exception exception) {
                    // empty catch block
                }
            }
        }
        return labels;
    }

    public static Collection<? extends PathObject> getSupportedObjects(Collection<? extends PathObject> availableObjects, Collection<Class<? extends PathObject>> supportedClasses) {
        return availableObjects.stream().filter(p -> supportedClasses.stream().anyMatch(s -> s.isInstance(p))).toList();
    }

    public static ROI getROI(PathObject pathObject, boolean preferNucleus) {
        ROI roi;
        if (preferNucleus && pathObject instanceof PathCellObject && (roi = ((PathCellObject)pathObject).getNucleusROI()) != null) {
            return roi;
        }
        return pathObject.getROI();
    }

    public static ROI getNucleusOrMainROI(PathObject pathObject) {
        ROI nucleus = PathObjectTools.getNucleusROI(pathObject);
        if (nucleus == null) {
            return pathObject.getROI();
        }
        return nucleus;
    }

    public static ROI getNucleusROI(PathObject pathObject) {
        if (pathObject instanceof PathCellObject) {
            PathCellObject cell = (PathCellObject)pathObject;
            return cell.getNucleusROI();
        }
        return null;
    }

    public static Collection<PathObject> getDescendantObjects(PathObject pathObject, Collection<PathObject> pathObjects, Class<? extends PathObject> cls) {
        if (pathObject == null || !pathObject.hasChildObjects()) {
            return pathObjects == null ? Collections.emptyList() : pathObjects;
        }
        if (pathObjects == null) {
            pathObjects = new ArrayList<PathObject>();
        }
        if (cls == null) {
            return pathObject.getDescendantObjects(pathObjects);
        }
        PathObjectTools.addPathObjectsRecursively(pathObject.getChildObjectsAsArray(), pathObjects, cls);
        return pathObjects;
    }

    private static void addPathObjectsRecursively(PathObject[] pathObjectsInput, Collection<PathObject> pathObjects, Class<? extends PathObject> cls) {
        for (PathObject childObject : pathObjectsInput) {
            if (cls == null || cls.isInstance(childObject)) {
                pathObjects.add(childObject);
            }
            if (!childObject.hasChildObjects()) continue;
            PathObjectTools.addPathObjectsRecursively(childObject.getChildObjectsAsArray(), pathObjects, cls);
        }
    }

    public static Map<PathObject, List<PathObject>> splitAreasByLines(Collection<? extends PathObject> pathObjects) {
        return PathObjectTools.splitAreasByLines(PathObjectTools.getLineObjects(pathObjects), PathObjectTools.getAreaObjects(pathObjects));
    }

    public static Map<PathObject, List<PathObject>> splitAreasByLines(Collection<? extends PathObject> areaObjects, Collection<? extends PathObject> lineObjects) {
        List<ROI> linesROIs = lineObjects.stream().map(PathObject::getROI).toList();
        if (linesROIs.isEmpty()) {
            logger.debug("No lines found to split!");
            return Collections.emptyMap();
        }
        LinkedHashMap<PathObject, List<PathObject>> map = new LinkedHashMap<PathObject, List<PathObject>>();
        for (PathObject pathObject : areaObjects) {
            ROI roi = pathObject.getROI();
            if (roi == null || !roi.isArea() || pathObject.isTMACore()) continue;
            List<Geometry> splitLineGeometries = linesROIs.stream().filter(line -> roi.getImagePlane().equals(line.getImagePlane())).map(ROI::getGeometry).toList();
            Geometry geom = roi.getGeometry();
            List<Geometry> split = GeometryTools.splitGeometryByLineStrings(geom, splitLineGeometries);
            if (split.size() == 1 && geom.equals(split.get(0))) continue;
            List<PathObject> results = split.stream().map(geometry -> GeometryTools.geometryToROI(geometry, roi.getImagePlane())).map(r -> PathObjectTools.createLike(pathObject, r)).toList();
            map.put(pathObject, results);
        }
        return map;
    }

    public static Map<PathObject, List<PathObject>> splitAreasByBufferedLines(Collection<? extends PathObject> pathObjects, double buffer) {
        return PathObjectTools.splitAreasByBufferedLines(PathObjectTools.getLineObjects(pathObjects), PathObjectTools.getAreaObjects(pathObjects), buffer);
    }

    public static Map<PathObject, List<PathObject>> splitAreasByBufferedLines(Collection<? extends PathObject> areaObjects, Collection<? extends PathObject> lineObjects, double buffer) {
        if (buffer <= 0.0) {
            return PathObjectTools.splitAreasByLines(areaObjects, lineObjects);
        }
        ROI[] lines = (ROI[])lineObjects.stream().map(PathObject::getROI).filter(ROI::isLine).map(r -> RoiTools.buffer(r, buffer)).toArray(ROI[]::new);
        if (lines.length == 0) {
            logger.debug("No lines found to split!");
            return Collections.emptyMap();
        }
        LinkedHashMap<PathObject, List<PathObject>> map = new LinkedHashMap<PathObject, List<PathObject>>();
        for (PathObject pathObject : areaObjects) {
            ROI roi = pathObject.getROI();
            if (roi == null || !roi.isArea() || pathObject.isTMACore()) continue;
            map.put(pathObject, PathObjectTools.splitObjectBySubtraction(pathObject, lines));
        }
        return map;
    }

    private static List<PathObject> splitObjectBySubtraction(PathObject pathObject, ROI ... roisToSubtract) {
        List<ROI> rois = PathObjectTools.splitROIBySubtraction(pathObject.getROI(), roisToSubtract);
        return rois.stream().map(roi -> PathObjectTools.createLike(pathObject, roi)).toList();
    }

    private static List<ROI> splitROIBySubtraction(ROI roi, ROI ... roisToSubtract) {
        if (roi == null) {
            return Collections.emptyList();
        }
        if (roisToSubtract.length == 0) {
            return Collections.singletonList(roi);
        }
        List<ROI> splitBefore = RoiTools.splitROI(roi);
        ROI roiSubtracted = RoiTools.subtract(roi, roisToSubtract);
        List<ROI> split = RoiTools.splitROI(roiSubtracted);
        if (split.size() == splitBefore.size()) {
            return Collections.singletonList(roi);
        }
        return split;
    }

    public static boolean mergePointsForSelectedObjectClasses(PathObjectHierarchy hierarchy) {
        Set pathClasses = hierarchy.getSelectionModel().getSelectedObjects().stream().filter(p -> p.isAnnotation() && p.getROI().isPoint()).map(p -> p.getPathClass()).collect(Collectors.toSet());
        boolean changes = false;
        for (PathClass pathClass : pathClasses) {
            changes = changes || PathObjectTools.mergePointsForClass(hierarchy, pathClass);
        }
        return changes;
    }

    public static boolean mergePointsForAllClasses(PathObjectHierarchy hierarchy) {
        if (hierarchy == null) {
            return false;
        }
        Set pathClasses = hierarchy.getAnnotationObjects().stream().filter(p -> p.getROI().isPoint()).map(p -> p.getPathClass()).collect(Collectors.toSet());
        boolean changes = false;
        for (PathClass pathClass : pathClasses) {
            changes = changes || PathObjectTools.mergePointsForClass(hierarchy, pathClass);
        }
        return changes;
    }

    public static boolean mergePointsForClass(PathObjectHierarchy hierarchy, PathClass pathClass) {
        Map<ImagePlane, List<PathObject>> map = hierarchy.getAnnotationObjects().stream().filter(p -> p.getROI().isPoint() && p.getPathClass() == pathClass).collect(Collectors.groupingBy(p -> p.getROI().getImagePlane()));
        ArrayList<PathObject> toRemove = new ArrayList<PathObject>();
        ArrayList<PathObject> toAdd = new ArrayList<PathObject>();
        for (Map.Entry<ImagePlane, List<PathObject>> entry : map.entrySet()) {
            List<PathObject> objectsToMerge = entry.getValue();
            if (objectsToMerge.size() <= 1) continue;
            ArrayList<Point2> pointsList = new ArrayList<Point2>();
            for (PathObject temp : objectsToMerge) {
                pointsList.addAll(((PointsROI)temp.getROI()).getAllPoints());
            }
            ROI points = ROIs.createPointsROI(pointsList, entry.getKey());
            toAdd.add(PathObjects.createAnnotationObject(points, pathClass));
            toRemove.addAll(objectsToMerge);
        }
        if (toAdd.isEmpty() && toRemove.isEmpty()) {
            return false;
        }
        hierarchy.removeObjects(toRemove, true);
        hierarchy.addObjects(toAdd);
        return true;
    }

    public static boolean standardizeClassifications(Collection<PathObject> pathObjects) {
        return PathObjectTools.standardizeClassifications(pathObjects, Comparator.naturalOrder());
    }

    public static boolean standardizeClassifications(Collection<PathObject> pathObjects, Comparator<String> comparator) {
        int nChanges = 0;
        HashMap<PathClass, PathClass> map = new HashMap<PathClass, PathClass>();
        for (PathObject pathObject : pathObjects) {
            PathClass pathClassNew;
            PathClass pathClass = pathObject.getPathClass();
            if (pathClass == null) continue;
            if (!map.containsKey(pathClass)) {
                pathClassNew = PathClassTools.sortNames(PathClassTools.uniqueNames(pathClass), comparator);
                map.put(pathClass, pathClassNew);
            } else {
                pathClassNew = (PathClass)map.get(pathClass);
            }
            if (pathClass.equals(pathClassNew)) continue;
            pathObject.setPathClass(pathClassNew);
            ++nChanges;
        }
        return nChanges > 0;
    }

    public static PathObject transformObject(PathObject pathObject, AffineTransform transform, boolean copyMeasurements) {
        return PathObjectTools.transformObject(pathObject, transform, copyMeasurements, false);
    }

    public static PathObject transformObject(PathObject pathObject, AffineTransform transform, boolean copyMeasurements, boolean createNewIDs) {
        return PathObjectTools.transformObjectImpl(pathObject, r -> PathObjectTools.maybeTransformROI(r, transform), copyMeasurements, createNewIDs);
    }

    private static PathObject transformObjectImpl(PathObject pathObject, Function<ROI, ROI> roiTransformer, boolean copyMeasurements, boolean createNewIDs) {
        PathObject newObject;
        ROI roi = roiTransformer.apply(pathObject.getROI());
        PathClass pathClass = pathObject.getPathClass();
        if (pathObject instanceof PathCellObject) {
            ROI roiNucleus = ((PathCellObject)pathObject).getNucleusROI();
            roiNucleus = roiNucleus == null ? null : roiTransformer.apply(roiNucleus);
            newObject = PathObjects.createCellObject(roi, roiNucleus, pathClass, null);
        } else if (pathObject instanceof PathTileObject) {
            newObject = PathObjects.createTileObject(roi, pathClass, null);
        } else if (pathObject instanceof PathDetectionObject) {
            newObject = PathObjects.createDetectionObject(roi, pathClass, null);
        } else if (pathObject instanceof PathAnnotationObject) {
            newObject = PathObjects.createAnnotationObject(roi, pathClass, null);
        } else if (pathObject instanceof PathRootObject) {
            newObject = new PathRootObject();
        } else if (pathObject instanceof TMACoreObject) {
            TMACoreObject core = (TMACoreObject)pathObject;
            newObject = PathObjects.createTMACoreObject(roi.getBoundsX(), roi.getBoundsY(), roi.getBoundsWidth(), roi.getBoundsHeight(), core.isMissing());
        } else {
            throw new UnsupportedOperationException("Unable to transform object " + String.valueOf(pathObject));
        }
        if (copyMeasurements && !pathObject.getMeasurementList().isEmpty()) {
            MeasurementList measurements = pathObject.getMeasurementList();
            newObject.getMeasurementList().putAll(measurements);
            newObject.getMeasurementList().close();
        }
        newObject.setName(pathObject.getName());
        newObject.setColor(pathObject.getColor());
        newObject.setLocked(pathObject.isLocked());
        if (copyMeasurements && !pathObject.getMetadata().isEmpty()) {
            newObject.getMetadata().putAll(pathObject.getMetadata());
        }
        if (!createNewIDs) {
            newObject.setID(pathObject.getID());
        }
        return newObject;
    }

    public static PathObject transformObjectRecursive(PathObject pathObject, AffineTransform transform, boolean copyMeasurements) {
        return PathObjectTools.transformObjectRecursive(pathObject, transform, true, true);
    }

    public static PathObject transformObjectRecursive(PathObject pathObject, AffineTransform transform, boolean copyMeasurements, boolean createNewIDs) {
        PathObject newObj = PathObjectTools.transformObject(pathObject, transform, copyMeasurements, createNewIDs);
        if (pathObject.hasChildObjects()) {
            List<PathObject> newChildObjects = pathObject.getChildObjects().parallelStream().map(p -> PathObjectTools.transformObjectRecursive(p, transform, copyMeasurements, createNewIDs)).toList();
            newObj.addChildObjects(newChildObjects);
        }
        return newObj;
    }

    private static ROI maybeTransformROI(ROI roi, AffineTransform transform) {
        if (roi == null || transform == null || transform.isIdentity()) {
            return roi;
        }
        return RoiTools.transformROI(roi, transform);
    }

    public static Map<String, PathObject> findByStringID(Collection<String> ids, Collection<? extends PathObject> pathObjects) {
        Map<String, PathObject> map = pathObjects.stream().collect(Collectors.toMap(p -> p.getID().toString(), p -> p));
        HashMap<String, PathObject> output = new HashMap<String, PathObject>();
        for (String id : ids) {
            output.put(id, map.getOrDefault(id, null));
        }
        return output;
    }

    public static Map<UUID, PathObject> findByUUID(Collection<UUID> ids, Collection<? extends PathObject> pathObjects) {
        Map<UUID, PathObject> map = pathObjects.stream().collect(Collectors.toMap(p -> p.getID(), p -> p));
        HashMap<UUID, PathObject> output = new HashMap<UUID, PathObject>();
        for (UUID id : ids) {
            output.put(id, map.getOrDefault(id, null));
        }
        return output;
    }

    public static Map<PathObject, PathObject> matchByID(Collection<? extends PathObject> sourceObjects, Collection<? extends PathObject> targetObjects) {
        Map<UUID, PathObject> map = targetObjects.stream().collect(Collectors.toMap(p -> p.getID(), p -> p));
        HashMap<PathObject, PathObject> output = new HashMap<PathObject, PathObject>();
        for (PathObject pathObject : sourceObjects) {
            output.put(pathObject, map.getOrDefault(pathObject.getID(), null));
        }
        return output;
    }

    public static boolean duplicateAllSelectedObjects(PathObjectHierarchy hierarchy) {
        return PathObjectTools.duplicateSelectedObjects(hierarchy, null);
    }

    public static boolean duplicateSelectedAnnotations(PathObjectHierarchy hierarchy) {
        return PathObjectTools.duplicateSelectedObjects(hierarchy, p -> p.isAnnotation());
    }

    public static boolean duplicateSelectedObjects(PathObjectHierarchy hierarchy, Predicate<PathObject> predicate) {
        if (predicate == null) {
            return PathObjectTools.duplicateObjects(hierarchy, new ArrayList<PathObject>(hierarchy.getSelectionModel().getSelectedObjects()));
        }
        List<PathObject> list = hierarchy.getSelectionModel().getSelectedObjects().stream().filter(predicate).toList();
        return PathObjectTools.duplicateObjects(hierarchy, list);
    }

    public static boolean duplicateObjects(PathObjectHierarchy hierarchy, Collection<PathObject> pathObjects) {
        return PathObjectTools.duplicateObjects(hierarchy, pathObjects, true);
    }

    public static boolean duplicateObjects(PathObjectHierarchy hierarchy, Collection<PathObject> pathObjects, boolean createNewIDs) {
        Map<PathObject, PathObject> map = pathObjects.stream().collect(Collectors.toMap(p -> p, p -> PathObjectTools.transformObject(p, null, true, createNewIDs)));
        if (map.isEmpty()) {
            logger.error("No selected objects to duplicate!");
            return false;
        }
        hierarchy.addObjects(map.values());
        if (hierarchy != null) {
            PathObject currentMainObject = hierarchy.getSelectionModel().getSelectedObject();
            hierarchy.fireHierarchyChangedEvent(PathObjectTools.class);
            hierarchy.getSelectionModel().setSelectedObjects(map.values(), map.getOrDefault(currentMainObject, null));
        }
        return true;
    }

    public static Collection<PathObject> removeOverlapsBySize(Collection<? extends PathObject> pathObjects, double overlapTolerance) {
        return PathObjectTools.removeOverlaps(pathObjects, Comparator.comparingDouble(p -> PathObjectTools.getROI(p, false).getArea()).reversed(), overlapTolerance);
    }

    public static Collection<PathObject> removeOverlapsByLocation(Collection<? extends PathObject> pathObjects, double overlapTolerance) {
        return PathObjectTools.removeOverlaps(pathObjects, Comparator.comparingDouble(p -> p.getROI().getBoundsY()).thenComparing(p -> p.getROI().getBoundsX()).thenComparing(p -> p.getROI().getCentroidY()).thenComparing(p -> p.getROI().getCentroidX()).thenComparing(p -> PathObjectTools.getROI(p, false).getArea()), overlapTolerance);
    }

    public static Collection<PathObject> removeOverlaps(Collection<? extends PathObject> pathObjects, Comparator<PathObject> comparator, double overlapTolerance) {
        Geometry geom;
        if (overlapTolerance != 0.0 && overlapTolerance <= -1.0) {
            logger.warn("A non-zero overlapTolerance <= -1.0 has no effect! Returning the same objects.");
            return new ArrayList<PathObject>(pathObjects);
        }
        LinkedHashSet<PathObject> output = new LinkedHashSet<PathObject>(pathObjects);
        ArrayList<PathObject> list = new ArrayList<PathObject>();
        for (PathObject pathObject : pathObjects) {
            if (!pathObject.hasROI() || !pathObject.getROI().isArea()) continue;
            list.add(pathObject);
        }
        Collections.sort(list, comparator);
        STRtree quadTree = new STRtree();
        HashMap<PathObject, Geometry> hashMap = new HashMap<PathObject, Geometry>();
        HashMap<PathObject, ImagePlane> planeMap = new HashMap<PathObject, ImagePlane>();
        for (PathObject pathObject : list) {
            ROI roi = PathObjectTools.getROI(pathObject, false);
            geom = roi.getGeometry();
            quadTree.insert(geom.getEnvelopeInternal(), (Object)pathObject);
            hashMap.put(pathObject, geom);
            planeMap.put(pathObject, roi.getImagePlane());
        }
        HashSet<PathObject> toRemove = new HashSet<PathObject>();
        for (PathObject pathObject : list) {
            if (toRemove.contains(pathObject)) continue;
            geom = (Geometry)hashMap.get(pathObject);
            ImagePlane plane = (ImagePlane)planeMap.get(pathObject);
            List potentialOverlaps = quadTree.query(geom.getEnvelopeInternal());
            for (PathObject p : potentialOverlaps) {
                Geometry geomP;
                if (p == pathObject || toRemove.contains(p)) continue;
                ImagePlane planeP = (ImagePlane)planeMap.get(p);
                if (plane.getZ() != planeP.getZ() || plane.getT() != planeP.getT() || !geom.intersects(geomP = (Geometry)hashMap.get(p)) || geom.touches(geomP)) continue;
                if (overlapTolerance != 0.0) {
                    try {
                        double overlap = geom.intersection(geomP).getArea();
                        double tol = overlapTolerance;
                        if (overlapTolerance < 0.0) {
                            tol = Math.min(geom.getArea(), geom.getArea()) * -overlapTolerance;
                        }
                        if (overlap < tol) {
                            continue;
                        }
                    }
                    catch (Exception e) {
                        logger.warn("Exception attempting to apply overlap tolerance: " + e.getLocalizedMessage(), (Throwable)e);
                    }
                }
                toRemove.add(p);
            }
        }
        output.removeAll(toRemove);
        return output;
    }

    public static PathObject mergeObjects(Collection<? extends PathObject> pathObjects) {
        PathObject result;
        if (pathObjects == null || pathObjects.isEmpty()) {
            throw new IllegalArgumentException("No objects provided to merge!");
        }
        PathObject first = pathObjects.iterator().next();
        if (pathObjects.size() == 1) {
            return first;
        }
        List<ROI> rois = pathObjects.stream().map(p -> p.getROI()).filter(r -> r != null && !r.isEmpty()).toList();
        ROI roi = RoiTools.union(rois);
        if (pathObjects.stream().allMatch(p -> p.isCell())) {
            List<ROI> nucleusRois = pathObjects.stream().map(p -> ((PathCellObject)p).getNucleusROI()).filter(r -> r != null && !r.isEmpty()).toList();
            ROI nucleusROI = RoiTools.union(nucleusRois);
            result = PathObjects.createCellObject(roi, nucleusROI, first.getPathClass(), null);
        } else if (pathObjects.stream().allMatch(p -> p.isAnnotation())) {
            result = PathObjects.createAnnotationObject(roi);
        } else if (pathObjects.stream().allMatch(p -> p.isTile())) {
            result = PathObjects.createTileObject(roi);
        } else if (pathObjects.stream().allMatch(p -> p.isDetection())) {
            result = PathObjects.createDetectionObject(roi);
        } else {
            throw new IllegalArgumentException("Unknow or mixed object types - cannot merge ROIs for " + String.valueOf(pathObjects));
        }
        result.setPathClass(first.getPathClass());
        result.setName(first.getName());
        return result;
    }

    public static <K> List<PathObject> mergeObjects(Collection<? extends PathObject> pathObjects, Function<? super PathObject, ? extends K> classifier) {
        Map<K, List<PathObject>> groups = pathObjects.stream().collect(Collectors.groupingBy(classifier));
        ArrayList<PathObject> output = new ArrayList<PathObject>();
        for (Map.Entry<K, List<PathObject>> entry : groups.entrySet()) {
            List<? super PathObject> group = entry.getValue();
            if (group.size() <= 1) {
                output.addAll(group);
                continue;
            }
            output.add(PathObjectTools.mergeObjects(group));
        }
        return output;
    }

    public static PathObject createLike(PathObject pathObject, ROI roiNew) {
        return PathObjectTools.createLike(pathObject, roiNew, null);
    }

    public static PathObject createLike(PathObject pathObject, ROI roiNew, ROI roiNucleus) {
        PathObject newObject;
        if (pathObject.isCell()) {
            newObject = PathObjects.createCellObject(roiNew, roiNucleus, pathObject.getPathClass(), null);
        } else if (pathObject.isAnnotation()) {
            newObject = PathObjects.createAnnotationObject(roiNew, pathObject.getPathClass());
        } else if (pathObject.isTile()) {
            newObject = PathObjects.createTileObject(roiNew, pathObject.getPathClass(), null);
        } else if (pathObject.isDetection()) {
            newObject = PathObjects.createDetectionObject(roiNew, pathObject.getPathClass());
        } else {
            throw new IllegalArgumentException("Unsupported object type - cannot create similar object for " + String.valueOf(pathObject));
        }
        newObject.setName(pathObject.getName());
        newObject.setColor(pathObject.getColor());
        return newObject;
    }

    private static void setSelectedObjectsLocked(PathObjectHierarchy hierarchy, Collection<? extends PathObject> pathObjects, boolean setToLocked) {
        ArrayList<PathObject> changed = new ArrayList<PathObject>();
        for (PathObject pathObject : pathObjects) {
            if (!(pathObject instanceof PathROIObject) || pathObject.isLocked() == setToLocked) continue;
            pathObject.setLocked(setToLocked);
            changed.add(pathObject);
        }
        if (hierarchy != null && !changed.isEmpty()) {
            hierarchy.fireObjectsChangedEvent(PathObjectTools.class, changed);
        }
    }

    public static void lockObjects(PathObjectHierarchy hierarchy, Collection<? extends PathObject> pathObjects) {
        PathObjectTools.setSelectedObjectsLocked(hierarchy, pathObjects, true);
    }

    public static void unlockObjects(PathObjectHierarchy hierarchy, Collection<? extends PathObject> pathObjects) {
        PathObjectTools.setSelectedObjectsLocked(hierarchy, pathObjects, false);
    }

    public static void lockSelectedObjects(PathObjectHierarchy hierarchy) {
        if (hierarchy == null) {
            return;
        }
        PathObjectTools.setSelectedObjectsLocked(hierarchy, hierarchy.getSelectionModel().getSelectedObjects(), true);
    }

    public static void unlockSelectedObjects(PathObjectHierarchy hierarchy) {
        if (hierarchy == null) {
            return;
        }
        PathObjectTools.setSelectedObjectsLocked(hierarchy, hierarchy.getSelectionModel().getSelectedObjects(), false);
    }

    public static void toggleSelectedObjectsLocked(PathObjectHierarchy hierarchy) {
        if (hierarchy == null) {
            return;
        }
        PathObjectTools.toggleObjectsLocked(hierarchy, hierarchy.getSelectionModel().getSelectedObjects());
    }

    public static void toggleObjectsLocked(PathObjectHierarchy hierarchy, Collection<? extends PathObject> pathObjects) {
        if (hierarchy == null) {
            return;
        }
        ArrayList<PathObject> changed = new ArrayList<PathObject>();
        for (PathObject pathObject : pathObjects) {
            if (!(pathObject instanceof PathROIObject)) continue;
            pathObject.setLocked(!pathObject.isLocked());
            changed.add(pathObject);
        }
        if (hierarchy != null && !changed.isEmpty()) {
            hierarchy.fireObjectsChangedEvent(PathObjectTools.class, changed);
        }
    }

    public static Set<String> getAvailableFeatures(Collection<? extends PathObject> pathObjects) {
        LinkedHashSet<String> featureSet = new LinkedHashSet<String>();
        List<String> lastNames = null;
        for (PathObject pathObject : pathObjects) {
            if (!pathObject.hasMeasurements()) continue;
            List<String> list = pathObject.getMeasurementList().getNames();
            if (lastNames != list) {
                featureSet.addAll(list);
            }
            lastNames = list;
        }
        return featureSet;
    }

    public static Map<PathObject, PathClass> createClassificationMap(Collection<? extends PathObject> pathObjects) {
        HashMap<PathObject, PathClass> mapPrevious = new HashMap<PathObject, PathClass>();
        for (PathObject pathObject : pathObjects) {
            mapPrevious.put(pathObject, pathObject.getPathClass());
        }
        return mapPrevious;
    }

    public static Collection<PathObject> restoreClassificationsFromMap(Map<PathObject, PathClass> classificationMap) {
        ArrayList<PathObject> changed = new ArrayList<PathObject>();
        for (Map.Entry<PathObject, PathClass> entry : classificationMap.entrySet()) {
            PathObject pathObject = entry.getKey();
            PathClass pathClass = entry.getValue();
            if (pathClass == PathClass.NULL_CLASS) {
                pathClass = null;
            }
            if (Objects.equals(pathObject.getPathClass(), pathClass)) continue;
            pathObject.setPathClass(pathClass);
            changed.add(pathObject);
        }
        return changed;
    }

    public static Set<PathClass> getRepresentedPathClasses(PathObjectHierarchy hierarchy, Class<? extends PathObject> cls) {
        LinkedHashSet<PathClass> pathClassSet = new LinkedHashSet<PathClass>();
        for (PathObject pathObject : hierarchy.getObjects(null, cls)) {
            if (pathObject.getPathClass() == null) continue;
            pathClassSet.add(pathObject.getPathClass());
        }
        return pathClassSet;
    }

    public static PathClass setIntensityClassification(PathObject pathObject, String measurementName, double ... thresholds) {
        boolean singleThreshold;
        if (thresholds.length == 0 || thresholds.length > 3) {
            throw new IllegalArgumentException("Between 1 and 3 intensity thresholds required!");
        }
        if (measurementName == null || measurementName.isEmpty()) {
            throw new IllegalArgumentException("Measurement name cannot be empty or null!");
        }
        PathClass baseClass = PathClassTools.getNonIntensityAncestorClass(pathObject.getPathClass());
        if (!PathClassTools.isNullClass(baseClass) && PathClassTools.isIgnoredClass(baseClass)) {
            return pathObject.getPathClass();
        }
        double intensityValue = pathObject.getMeasurementList().get(measurementName);
        boolean bl = singleThreshold = thresholds.length == 1;
        if (Double.isNaN(intensityValue)) {
            pathObject.setPathClass(baseClass);
        } else if (intensityValue < thresholds[0]) {
            pathObject.setPathClass(PathClass.getNegative(baseClass));
        } else if (singleThreshold) {
            pathObject.setPathClass(PathClass.getPositive(baseClass));
        } else if (thresholds.length >= 3 && intensityValue >= thresholds[2]) {
            pathObject.setPathClass(PathClass.getThreePlus(baseClass));
        } else if (thresholds.length >= 2 && intensityValue >= thresholds[1]) {
            pathObject.setPathClass(PathClass.getTwoPlus(baseClass));
        } else if (intensityValue >= thresholds[0]) {
            pathObject.setPathClass(PathClass.getOnePlus(baseClass));
        }
        return pathObject.getPathClass();
    }

    public static void setIntensityClassifications(Collection<? extends PathObject> pathObjects, String measurementName, double ... thresholds) {
        pathObjects.stream().forEach(p -> PathObjectTools.setIntensityClassification(p, measurementName, thresholds));
    }

    private static class ImageRegionPredicate
    implements Predicate<PathObject> {
        private final ImageRegion region;
        private final PreparedGeometry geometry;

        ImageRegionPredicate(ImageRegion region) {
            this.region = region;
            this.geometry = PreparedGeometryFactory.prepare((Geometry)ROIs.createRectangleROI(region).getGeometry());
        }

        @Override
        public boolean test(PathObject p) {
            return p.hasROI() && this.region.intersects(ImageRegion.createInstance(p.getROI())) && this.geometry.intersects(p.getROI().getGeometry());
        }
    }
}

