/*
 * Decompiled with CFR 0.152.
 */
package qupath.lib.objects.utils;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.locationtech.jts.geom.Envelope;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.Polygon;
import org.locationtech.jts.geom.TopologyException;
import org.locationtech.jts.geom.prep.PreparedGeometry;
import org.locationtech.jts.geom.prep.PreparedGeometryFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import qupath.lib.objects.PathObject;
import qupath.lib.objects.PathObjects;
import qupath.lib.roi.GeometryTools;
import qupath.lib.roi.interfaces.ROI;

public class Tiler {
    private static final Logger logger = LoggerFactory.getLogger(Tiler.class);
    private final int tileWidth;
    private final int tileHeight;
    private final boolean cropToParent;
    private final boolean filterByCentroid;
    private final TileAlignment alignment;

    private Tiler(int tileWidth, int tileHeight, boolean cropToParent, TileAlignment alignment, boolean filterByCentroid) {
        if (tileWidth <= 0 || tileHeight <= 0) {
            throw new IllegalArgumentException("tileWidth and tileHeight must be > 0, but were " + tileWidth + " and " + tileHeight);
        }
        this.tileWidth = tileWidth;
        this.tileHeight = tileHeight;
        this.cropToParent = cropToParent;
        this.alignment = alignment;
        this.filterByCentroid = filterByCentroid;
    }

    public int getTileWidth() {
        return this.tileWidth;
    }

    public int getTileHeight() {
        return this.tileHeight;
    }

    public boolean getCropToParent() {
        return this.cropToParent;
    }

    public TileAlignment getAlignment() {
        return this.alignment;
    }

    public boolean getFilterByCentroid() {
        return this.filterByCentroid;
    }

    public List<Geometry> createGeometries(Geometry parent) {
        if (parent == null) {
            logger.warn("Tiler.createGeometries() called with null parent - no tiles will be created");
            return new ArrayList<Geometry>();
        }
        Envelope boundingBox = parent.getEnvelopeInternal();
        double xStart = boundingBox.getMinX();
        double xEnd = boundingBox.getMaxX();
        switch (this.alignment.ordinal()) {
            case 0: 
            case 3: 
            case 6: {
                break;
            }
            case 1: 
            case 4: 
            case 7: {
                double bBoxWidth = xEnd - xStart;
                if (this.filterByCentroid) {
                    xStart += Tiler.calculateInteriorOffset(this.tileWidth, bBoxWidth);
                    break;
                }
                xStart += Tiler.calculateExteriorOffset(this.tileWidth, bBoxWidth);
                break;
            }
            case 2: 
            case 5: 
            case 8: {
                xStart += Tiler.calculateRightAlignedStartOffset(this.tileWidth, xEnd - xStart);
            }
        }
        double yStart = boundingBox.getMinY();
        double yEnd = boundingBox.getMaxY();
        switch (this.alignment.ordinal()) {
            case 0: 
            case 1: 
            case 2: {
                break;
            }
            case 3: 
            case 4: 
            case 5: {
                double bBoxHeight = yEnd - yStart;
                if (this.filterByCentroid) {
                    yStart += Tiler.calculateInteriorOffset(this.tileHeight, bBoxHeight);
                    break;
                }
                yStart += Tiler.calculateExteriorOffset(this.tileHeight, bBoxHeight);
                break;
            }
            case 6: 
            case 7: 
            case 8: {
                yStart += Tiler.calculateRightAlignedStartOffset(this.tileHeight, yEnd - yStart);
            }
        }
        ArrayList<Polygon> tiles = new ArrayList<Polygon>();
        int x = (int)xStart;
        while ((double)x < xEnd) {
            int y = (int)yStart;
            while ((double)y < yEnd) {
                tiles.add(GeometryTools.createRectangle(x, y, this.tileWidth, this.tileHeight));
                y += this.tileHeight;
            }
            x += this.tileWidth;
        }
        PreparedGeometry preparedParent = PreparedGeometryFactory.prepare((Geometry)parent);
        return tiles.parallelStream().map(Tiler.createTileFilter(preparedParent, this.cropToParent, this.filterByCentroid)).filter(Objects::nonNull).collect(Collectors.toList());
    }

    private static Function<Geometry, Geometry> createTileFilter(PreparedGeometry parent, boolean cropToParent, boolean filterByCentroid) {
        return tile -> {
            if (!parent.intersects(tile)) {
                return null;
            }
            if (parent.covers(tile)) {
                return tile;
            }
            if (cropToParent) {
                try {
                    return tile.intersection(parent.getGeometry());
                }
                catch (TopologyException e) {
                    logger.warn("Exception calculating tile intersection - tile will be skipped", (Throwable)e);
                    return null;
                }
            }
            if (!filterByCentroid || parent.contains((Geometry)tile.getCentroid())) {
                return tile;
            }
            return null;
        };
    }

    public List<ROI> createROIs(ROI parent) {
        return this.createGeometries(parent.getGeometry()).stream().map(g -> GeometryTools.geometryToROI(g, parent.getImagePlane())).collect(Collectors.toList());
    }

    public List<PathObject> createObjects(ROI parent, Function<ROI, PathObject> creator) {
        return this.createROIs(parent).stream().map(creator).collect(Collectors.toList());
    }

    public List<PathObject> createTiles(ROI parent) {
        return this.createObjects(parent, PathObjects::createTileObject);
    }

    public List<PathObject> createAnnotations(ROI parent) {
        return this.createObjects(parent, PathObjects::createAnnotationObject);
    }

    private static double calculateRightAlignedStartOffset(int tileDim, double parentDim) {
        double mod = parentDim % (double)tileDim;
        if (mod == 0.0) {
            return 0.0;
        }
        return -((double)tileDim - mod);
    }

    private static double calculateInteriorOffset(int tileDim, double parentDim) {
        double mod = parentDim % (double)tileDim;
        if (mod == 0.0) {
            return 0.0;
        }
        return mod / 2.0;
    }

    private static double calculateExteriorOffset(int tileDim, double parentDim) {
        double mod = parentDim % (double)tileDim;
        if (mod == 0.0) {
            return 0.0;
        }
        return -((double)tileDim - mod) / 2.0;
    }

    public static Builder builder(int tileSize) {
        return Tiler.builder(tileSize, tileSize);
    }

    public static Builder builder(int tileWidth, int tileHeight) {
        return new Builder(tileWidth, tileHeight);
    }

    public static Builder builder(Tiler tiler) {
        return new Builder(tiler);
    }

    public static enum TileAlignment {
        TOP_LEFT,
        TOP_CENTER,
        TOP_RIGHT,
        CENTER_LEFT,
        CENTER,
        CENTER_RIGHT,
        BOTTOM_LEFT,
        BOTTOM_CENTER,
        BOTTOM_RIGHT;

    }

    public static class Builder {
        private int tileWidth;
        private int tileHeight;
        private boolean cropToParent = true;
        private TileAlignment alignment = TileAlignment.CENTER;
        private boolean filterByCentroid = false;

        private Builder(int tileWidth, int tileHeight) {
            this.tileWidth = tileWidth;
            this.tileHeight = tileHeight;
        }

        private Builder(Tiler tiler) {
            this.tileWidth = tiler.tileWidth;
            this.tileHeight = tiler.tileHeight;
            this.cropToParent = tiler.cropToParent;
            this.alignment = tiler.alignment;
            this.filterByCentroid = tiler.filterByCentroid;
        }

        public Builder tileHeight(int tileHeight) {
            this.tileHeight = tileHeight;
            return this;
        }

        public Builder tileWidth(int tileWidth) {
            this.tileWidth = tileWidth;
            return this;
        }

        public Builder cropTiles(boolean cropToParent) {
            this.cropToParent = cropToParent;
            return this;
        }

        public Builder alignment(TileAlignment alignment) {
            this.alignment = alignment;
            return this;
        }

        public Builder alignTopLeft() {
            return this.alignment(TileAlignment.TOP_LEFT);
        }

        public Builder alignTopCenter() {
            return this.alignment(TileAlignment.TOP_CENTER);
        }

        public Builder alignTopRight() {
            return this.alignment(TileAlignment.TOP_RIGHT);
        }

        public Builder alignCenterLeft() {
            return this.alignment(TileAlignment.CENTER_LEFT);
        }

        public Builder alignCenter() {
            return this.alignment(TileAlignment.CENTER);
        }

        public Builder alignCenterRight() {
            return this.alignment(TileAlignment.CENTER_RIGHT);
        }

        public Builder alignBottomLeft() {
            return this.alignment(TileAlignment.BOTTOM_LEFT);
        }

        public Builder alignBottomCenter() {
            return this.alignment(TileAlignment.BOTTOM_CENTER);
        }

        public Builder alignBottomRight() {
            return this.alignment(TileAlignment.BOTTOM_RIGHT);
        }

        public Builder filterByCentroid(boolean filterByCentroid) {
            this.filterByCentroid = filterByCentroid;
            return this;
        }

        public Tiler build() {
            return new Tiler(this.tileWidth, this.tileHeight, this.cropToParent, this.alignment, this.filterByCentroid);
        }
    }
}

