/*
 * Decompiled with CFR 0.152.
 */
package qupath.lib.images.writers.ome;

import java.awt.image.BufferedImage;
import java.io.File;
import java.net.URI;
import java.util.Arrays;
import java.util.Map;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.DoubleStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import picocli.CommandLine;
import qupath.lib.common.GeneralTools;
import qupath.lib.gui.extensions.Subcommand;
import qupath.lib.gui.images.stores.DefaultImageRegionStore;
import qupath.lib.gui.images.stores.ImageRegionStoreFactory;
import qupath.lib.gui.prefs.PathPrefs;
import qupath.lib.images.servers.ImageServer;
import qupath.lib.images.servers.ImageServerProvider;
import qupath.lib.images.servers.ImageServers;
import qupath.lib.images.servers.bioformats.BioFormatsServerBuilder;
import qupath.lib.images.writers.ome.OMEPyramidWriter;
import qupath.lib.images.writers.ome.zarr.OMEZarrWriter;
import qupath.lib.regions.ImageRegion;

@CommandLine.Command(name="convert-ome", description={"Converts an input image to OME-TIFF or OME-Zarr."}, sortOptions=false)
public class ConvertCommand
implements Runnable,
Subcommand {
    private static final Logger logger = LoggerFactory.getLogger(ConvertCommand.class);
    @CommandLine.Parameters(index="0", paramLabel="input", description={"Path of the file to convert."})
    private File inputFile;
    @CommandLine.Parameters(index="1", paramLabel="output", description={"Path of the output file.", "The extension of the file must be .ome.zarr to create a Zarr file", "or .ome.tiff to create an OME TIFF file."})
    private File outputFile;
    @CommandLine.Option(names={"-r", "--crop"}, defaultValue="", description={"Bounding box to crop the input image.", "Defined in terms of full-resolution pixel coordinates in the form x,y,w,h.", "If empty (default), the full image will be exported."})
    private String crop;
    @CommandLine.Option(names={"-z", "--zslices"}, defaultValue="all", description={"Request which z-slice(s) will be exported.", "Value may be \"all\" (the default), a single number (e.g. \"1\") or a range (e.g. \"1-5\"). Indices are 1-based and ranges are inclusive."})
    private String zSlices;
    @CommandLine.Option(names={"-t", "--timepoints"}, defaultValue="all", description={"Request which timepoints will be exported.", "Value may be \"all\" (the default), a single number (e.g. \"1\") or a range (e.g. \"1-5\"). Indices are 1-based and ranges are inclusive."})
    private String timepoints;
    @CommandLine.Option(names={"-d", "--downsample"}, defaultValue="1.0", description={"Downsample the input image by the given factor (default=1)."})
    private double downsample;
    @CommandLine.Option(names={"-y", "--pyramid-scale"}, defaultValue="1.0", description={"Scale factor for pyramidal images.", "Each pyramidal level is scaled down by the specified factor (> 1).", "The downsamples of the original image are used if <= 1."})
    private double pyramid;
    @CommandLine.Option(names={"--big-tiff"}, defaultValue="_NULL_", description={"Request to write a big tiff, which is required when writing a TIFF images > 4GB.", "Default is to automatically decide based on image size. Choose --big-tiff=false to force a non-big-tiff to be written.", "Only relevant for TIFF files."})
    private Boolean bigTiff;
    @CommandLine.Option(names={"--tile-size"}, defaultValue="-1", description={"Set the tile size (of equal height and width)."})
    private int tileSize;
    @CommandLine.Option(names={"--tile-width"}, defaultValue="512", description={"Set the tile width (default=512)."})
    private int tileWidth;
    @CommandLine.Option(names={"--tile-height"}, defaultValue="512", description={"Set the tile height (default=512)."})
    private int tileHeight;
    @CommandLine.Option(names={"-c", "--compression"}, defaultValue="DEFAULT", description={"Type of compression to use for writing TIFF files.", "Only relevant for TIFF files", "Options: ${COMPLETION-CANDIDATES}"})
    private OMEPyramidWriter.CompressionType compression;
    @CommandLine.Option(names={"-p", "--parallelize"}, defaultValue="true", paramLabel="parallelization", description={"Parallelize tile export if possible (default=true)."}, negatable=true)
    private boolean parallelize;
    @CommandLine.Option(names={"--overwrite"}, defaultValue="false", description={"Overwrite any existing file with the same name as the output (default=false)."})
    private boolean overwrite = false;
    @CommandLine.Option(names={"--series"}, description={"Series number.", "Setting this will ensure the image is opened using Bio-Formats and control which image is read from the file.", "If it is not specified, the default image will be read (typically series 0)."})
    private int series = -1;
    @CommandLine.Option(names={"-h", "--help"}, usageHelp=true, description={"Show this help message and exit."})
    private boolean usageHelpRequested;

    @Override
    public void run() {
        OutputType outputType;
        long startTime = System.currentTimeMillis();
        if (this.inputFile == null || this.outputFile == null) {
            logger.error("Incorrect given path(s)");
            System.exit(-1);
        }
        OutputType outputType2 = outputType = this.outputFile.getAbsolutePath().toLowerCase().endsWith(OutputType.ZARR.getExtension()) ? OutputType.ZARR : OutputType.TIFF;
        if (!this.outputFile.getAbsolutePath().toLowerCase().endsWith(outputType.getExtension())) {
            this.outputFile = new File(this.outputFile.getParentFile(), GeneralTools.getNameWithoutExtension((File)this.outputFile) + outputType.getExtension());
        }
        if (this.outputFile.exists() && !this.overwrite) {
            logger.error("Output file " + String.valueOf(this.outputFile) + " exists!");
            System.exit(-1);
        }
        if (this.inputFile.equals(this.outputFile)) {
            logger.error("Input and output files are the same!");
            System.exit(-1);
        }
        String[] args = this.series >= 0 ? new String[]{"--classname", BioFormatsServerBuilder.class.getName(), "--series", Integer.toString(this.series)} : new String[]{};
        this.createTileCache();
        try (ImageServer server = ImageServers.buildServer((URI)this.inputFile.toURI(), (String[])args);){
            Range timepointsRange;
            Range zSlicesRange;
            if (this.tileSize > -1) {
                this.tileWidth = this.tileSize;
                this.tileHeight = this.tileSize;
            }
            if (this.downsample < 1.0) {
                this.downsample = server.getDownsampleForResolution(0);
            }
            if (!ConvertCommand.isValidRange(zSlicesRange = ConvertCommand.getRange(this.zSlices, server.nZSlices(), "zslices"), server.nZSlices())) {
                logger.error("Invalid range of --zslices: {}, image supports {}-{}", new Object[]{this.zSlices, 1, server.nZSlices()});
                System.exit(-1);
            }
            if (!ConvertCommand.isValidRange(timepointsRange = ConvertCommand.getRange(this.timepoints, server.nTimepoints(), "timepoints"), server.nTimepoints())) {
                logger.error("Invalid range of --timepoints: {}, image supports {}-{}", new Object[]{this.timepoints, 1, server.nTimepoints()});
                System.exit(-1);
            }
            Optional<ImageRegion> boundingBox = ConvertCommand.getBoundingBox(this.crop);
            switch (outputType.ordinal()) {
                case 0: {
                    if (!this.compression.supportsImage(server)) {
                        logger.error("Chosen compression " + this.compression.toString() + " is not compatible with the input image.");
                        System.exit(-1);
                    }
                    Object builder = new OMEPyramidWriter.Builder((ImageServer<BufferedImage>)server).compression(this.compression).tileSize(this.tileWidth, this.tileHeight).parallelize(this.parallelize).zSlices(zSlicesRange.start(), zSlicesRange.end()).timePoints(timepointsRange.start(), timepointsRange.end());
                    if (this.bigTiff != null) {
                        builder = ((OMEPyramidWriter.Builder)builder).bigTiff(this.bigTiff);
                    }
                    if (this.pyramid > 1.0) {
                        ((OMEPyramidWriter.Builder)builder).scaledDownsampling(this.downsample, this.pyramid);
                    } else {
                        ((OMEPyramidWriter.Builder)builder).downsamples(DoubleStream.concat(DoubleStream.of(this.downsample), Arrays.stream(server.getPreferredDownsamples()).filter(d -> d > this.downsample)).toArray());
                    }
                    if (boundingBox.isPresent()) {
                        ((OMEPyramidWriter.Builder)builder).region(boundingBox.get().getX(), boundingBox.get().getY(), boundingBox.get().getWidth(), boundingBox.get().getHeight());
                    }
                    ((OMEPyramidWriter.Builder)builder).build().writeSeries(this.outputFile.getAbsolutePath());
                    break;
                }
                case 1: {
                    Object builder = new OMEZarrWriter.Builder((ImageServer<BufferedImage>)server, this.outputFile.getAbsolutePath()).setTileWidth(this.tileWidth).setTileHeight(this.tileHeight).setBoundingBox(boundingBox.orElse(null)).setZSlices(zSlicesRange.start(), zSlicesRange.end()).setTimepoints(timepointsRange.start(), timepointsRange.end());
                    if (!this.parallelize) {
                        ((OMEZarrWriter.Builder)builder).setNumberOfThreads(1);
                    }
                    if (this.pyramid > 1.0) {
                        ((OMEZarrWriter.Builder)builder).setDownsamples(DoubleStream.iterate(this.downsample, d -> (int)((double)server.getWidth() / d) > this.tileWidth && (int)((double)server.getHeight() / d) > this.tileHeight, d -> d * this.pyramid).toArray());
                    } else {
                        ((OMEZarrWriter.Builder)builder).setDownsamples(DoubleStream.concat(DoubleStream.of(this.downsample), Arrays.stream(server.getPreferredDownsamples()).filter(d -> d > this.downsample)).toArray());
                    }
                    OMEZarrWriter writer = ((OMEZarrWriter.Builder)builder).build();
                    try {
                        writer.writeImage();
                        if (writer == null) break;
                    }
                    catch (Throwable throwable) {
                        if (writer != null) {
                            try {
                                writer.close();
                            }
                            catch (Throwable throwable2) {
                                throwable.addSuppressed(throwable2);
                            }
                        }
                        throw throwable;
                    }
                    writer.close();
                    break;
                }
            }
            long duration = System.currentTimeMillis() - startTime;
            logger.info(String.format("%s written in %.1f seconds", this.outputFile.getAbsolutePath(), (double)duration / 1000.0));
        }
        catch (Exception e) {
            logger.error("Error while creating image", (Throwable)e);
            System.exit(-1);
        }
    }

    private void createTileCache() {
        double percentage;
        Runtime rt = Runtime.getRuntime();
        long maxAvailable = rt.maxMemory();
        if (maxAvailable == Long.MAX_VALUE) {
            logger.warn("No inherent maximum memory set - for caching purposes, will assume 64 GB");
            maxAvailable = 0x1000000000L;
        }
        if ((percentage = PathPrefs.tileCachePercentageProperty().get()) < 10.0) {
            percentage = 10.0;
        } else if (percentage > 90.0) {
            percentage = 90.0;
        }
        long tileCacheSize = Math.round((double)maxAvailable * (percentage / 100.0));
        logger.info(String.format("Setting tile cache size to %.2f MB (%.1f%% max memory)", (double)tileCacheSize / 1048576.0, percentage));
        DefaultImageRegionStore imageRegionStore = ImageRegionStoreFactory.createImageRegionStore((long)tileCacheSize);
        ImageServerProvider.setCache((Map)imageRegionStore.getCache(), BufferedImage.class);
    }

    private static Range getRange(String rangeText, int maxRange, String rangeLabel) {
        String patternRange = "(\\d+)-(\\d+)";
        String patternInteger = "\\d+";
        if (rangeText == null || rangeText.isBlank() || "all".equals(rangeText)) {
            return new Range(0, maxRange);
        }
        if (rangeText.matches(patternRange)) {
            int end;
            int start = Integer.parseInt(rangeText.substring(0, rangeText.indexOf("-")));
            if (start == (end = Integer.parseInt(rangeText.substring(rangeText.indexOf("-") + 1)))) {
                return new Range(start - 1, start);
            }
            if (start > end) {
                logger.error(String.format("Invalid range of --%s (must be ascending): %s", rangeLabel, rangeText));
                System.exit(-1);
                return null;
            }
            return new Range(start - 1, end);
        }
        if (rangeText.matches(patternInteger)) {
            int v = Integer.parseInt(rangeText);
            return new Range(v - 1, v);
        }
        logger.error(String.format("Unknown value for --%s: %s", rangeLabel, rangeText));
        System.exit(-1);
        return null;
    }

    private static boolean isValidRange(Range range, int maxRange) {
        return range.start() >= 0 && range.end() <= maxRange && range.start() < range.end();
    }

    private static Optional<ImageRegion> getBoundingBox(String crop) {
        if (crop != null && !crop.isBlank()) {
            Matcher matcher = Pattern.compile("(\\d+),(\\d+),(\\d+),(\\d+)").matcher(crop);
            if (matcher.matches()) {
                int x = Integer.parseInt(matcher.group(1));
                int y = Integer.parseInt(matcher.group(2));
                int w = Integer.parseInt(matcher.group(3));
                int h = Integer.parseInt(matcher.group(4));
                return Optional.of(ImageRegion.createInstance((int)x, (int)y, (int)w, (int)h, (int)0, (int)0));
            }
            logger.error("Unknown value for --crop: " + crop);
            System.exit(-1);
            return Optional.empty();
        }
        return Optional.empty();
    }

    private static enum OutputType {
        TIFF(".ome.tif"),
        ZARR(".ome.zarr");

        private final String extension;

        private OutputType(String extension) {
            this.extension = extension;
        }

        public String getExtension() {
            return this.extension;
        }
    }

    private record Range(int start, int end) {
    }
}

